package util

import (
	"reflect"
	"testing"

	"google.golang.org/protobuf/types/known/anypb"
	structpb "google.golang.org/protobuf/types/known/structpb"

	fcontext "code.justin.tv/amzn/TwitchFeatureContext/schema/TwitchFeatureContext"
)

func TestMarshalMLFSContext(t *testing.T) {
	// Explicitly declared
	mlfs_context := fcontext.MLFSContext{
		Features: &structpb.Struct{
			Fields: map[string]*structpb.Value{
				"feature_1": &structpb.Value{
					Kind: &structpb.Value_StringValue{
						StringValue: "test_value",
					},
				},
				"feature_2": &structpb.Value{
					Kind: &structpb.Value_NumberValue{
						NumberValue: 12345,
					},
				},
			},
		},
	}

	_, err := MarshalMLFSContext(&mlfs_context)
	if err != nil {
		t.Error(err)
	}

	// Using structpb.NewStruct()
	m := map[string]interface{}{
		"feature1": "value1",
		"age":      23,
	}

	mlfs_struct, err := structpb.NewStruct(m)
	if err != nil {
		t.Error(err)
	}
	mlfs_context = fcontext.MLFSContext{
		Features: mlfs_struct,
	}
	_, err = MarshalMLFSContext(&mlfs_context)
	if err != nil {
		t.Error(err)
	}
}

func TestUnmarshalMLFSContext(t *testing.T) {
	mlfs_context := fcontext.MLFSContext{
		Features: &structpb.Struct{
			Fields: map[string]*structpb.Value{
				"feature_1": &structpb.Value{
					Kind: &structpb.Value_StringValue{
						StringValue: "test_value",
					},
				},
				"feature_2": &structpb.Value{
					Kind: &structpb.Value_NumberValue{
						NumberValue: 12345,
					},
				},
			},
		},
	}

	m, err := anypb.New(&mlfs_context)
	if err != nil {
		t.Error(err)
	}

	unmarshaled_mlfs, err := UnmarshalMLFSContext(m)
	if err != nil {
		t.Error(err)
	}

	if !reflect.DeepEqual(unmarshaled_mlfs.Features.AsMap(), mlfs_context.Features.AsMap()) {
		t.Errorf("UnmarshalMLFSContext.Features content did not match original struct: %#v ----------- %#v", unmarshaled_mlfs.Features.AsMap(), mlfs_context.Features.AsMap())
	}
}
