package clients

import (
	"context"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	ddb "github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit/v3"
)

type DDBSubset interface {
	BatchGetItemWithContext(ctx aws.Context, input *ddb.BatchGetItemInput, opts ...request.Option) (*ddb.BatchGetItemOutput, error)
}

var _ DDBSubset = (*ddb.DynamoDB)(nil)

type ddbClient struct {
	client          DDBSubset
	batchGetCircuit *circuit.Circuit
}

func NewDDBClient(sess *session.Session, config *aws.Config, manager *circuit.Manager, circuitConfig circuit.Config, id string) DDBSubset {
	return &ddbClient{
		client:          ddb.New(sess, config),
		batchGetCircuit: manager.MustCreateCircuit(fmt.Sprintf("mlfs:%s", id), circuitConfig),
	}
}

func (d *ddbClient) BatchGetItemWithContext(ctx aws.Context, input *ddb.BatchGetItemInput, opts ...request.Option) (*ddb.BatchGetItemOutput, error) {
	var out *ddb.BatchGetItemOutput
	var err, innerErr error
	err = d.batchGetCircuit.Run(ctx, func(ctx context.Context) error {
		out, innerErr = d.client.BatchGetItemWithContext(ctx, input, opts...)
		return innerErr
	})
	if err != nil {
		return nil, fmt.Errorf("failed to call BatchGetitem: %w", err)
	}
	return out, nil
}
