package fsc

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/cep21/circuit/v3"

	boundedCache "code.justin.tv/amzn/TwitchBoundedCache"
	logging "code.justin.tv/amzn/TwitchLogging"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"

	"code.justin.tv/amzn/TwitchFeatureStoreClient/cache"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/metadata"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/metrics"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/source"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/store"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

const FeatureMetadataBucketBeta = "mlfs-feature-metadata-beta"
const FeatureMetadataBucketProd = "mlfs-feature-metadata-prod"

type ENV string

const (
	Beta          ENV           = "Beta"
	Prod          ENV           = "Prod"
	defaultJitter time.Duration = 10 * time.Minute
)

type FeatureStoreClient interface {
	// GetDefaultFeatureSourceBuilder returns a default FeatureSourceBuilder implementation
	// which supports fetching features stored in MLFS.
	GetDefaultFeatureSourceBuilder() source.FeatureSourceBuilder

	// GetFeatureMetadata returns a read-only feature metadata provider, which is based
	// on metadata defined in the Feature Registry
	GetFeatureMetadata(key types.FeatureKey) (metadata.Provider, error)
}

type Client struct {
	// Used only for debugging purpose
	Logger      logging.Logger
	Config      client.ConfigProvider
	Region      string
	Environment ENV
	ServiceName string

	// An optional identifier used for services that have multiple feature store clients
	ClientIdentifier string

	// An optional http client which will be used for fetching feature from OFS.
	// Default http client is used if it is not specified.
	HttpClient *http.Client

	// The Observer is use to report high-level feature utilization metrics.
	// Metrics are disabled if Observer is not specified.
	Observer telemetry.SampleObserver

	// The config will apply to all circuits used to talk to OFS. The metrics
	// collector needs to be configured in the CircuitConfig or CircuitManager,
	// otherwise no circuit metrics will be reported.
	// Circuits with "mlfs:${account id}:${table name}" will be created inside FSC.
	// Make sure you do not have conflict circuit names inside the circuit manager.
	CircuitConfig  circuit.Config
	CircuitManager *circuit.Manager

	// Deprecated. Use Cache to provide a cache functionality.
	// To keep the old behavior, use cache.NewTwitchBoundedCache(...).
	// You will however be responsible to call Stop on the clock If your system requires it.
	// Stopping the clock used to be handled by the Client.
	// Create a boundedCache config object to enable caching on the feature store client.
	// by default, FSC will used a cached background call to reduce the number of calls
	// to time.Now() which adds CPU overheads on Fargate
	// This is ignored if Cache is provided.
	CacheConfig boundedCache.Config

	// Cache used to cache features with `FeatureParams.CacheFeature` set to `true`.
	// An in-memory implementation using TwitchBoundedCache is provided and can be constructed
	// with `cache.NewTwitchBoundedCache(...)`
	Cache cache.Cache

	featuresToCache    map[types.FeatureKey]types.FeatureParams              // maps feature key to ttl in Cache
	metadata           map[types.FeatureKey]metadata.Provider                // maps feature key to metadata
	featureSources     map[types.FeatureKey]source.IdentifiableFeatureSource // maps feature key to feature source
	identifiableSource map[string]source.IdentifiableFeatureSource
	// Deprecated. This is ignored if Cache is provided.
	cachedclock *boundedCache.CachedClock
}

var _ FeatureStoreClient = &Client{}

//Load feature metadata from feature registry
func (c *Client) loadMetadata(ctx context.Context, features []types.FeatureKey, manager *circuit.Manager) error {
	s3Client := s3.New(c.Config, aws.NewConfig().WithRegion(c.Region))
	bucket := FeatureMetadataBucketBeta
	if c.Environment == Prod {
		bucket = FeatureMetadataBucketProd
	}
	metadataStoreCircuitName := "mlfs_metadata"
	if c.ClientIdentifier != "" {
		metadataStoreCircuitName = fmt.Sprintf("%s:%s", metadataStoreCircuitName, c.ClientIdentifier)
	}
	metadataStore := &store.S3Store{
		S3:     s3Client,
		Bucket: bucket,
		Circuits: store.S3Circuits{ // TODO remove this circuit
			GetCircuit: manager.MustCreateCircuit(metadataStoreCircuitName, circuit.Config{
				Execution: circuit.ExecutionConfig{
					Timeout: 2 * time.Minute, // override the timeout for reading feature metadata
				},
				Metrics: c.CircuitConfig.Metrics,
			}),
		},
	}

	m, err := metadata.LoadMetadata(ctx, metadataStore, features)
	if err != nil {
		return err
	}
	c.metadata = m
	return nil
}

func (c *Client) InitializeFeatures(ctx context.Context, features ...types.FeatureParams) error {
	manager := &circuit.Manager{}
	if c.CircuitManager != nil {
		manager = c.CircuitManager
	}

	if c.Logger == nil {
		c.Logger = &types.EmptyLogger{}
	}

	if len(c.ServiceName) == 0 || len(c.Region) == 0 || len(c.Environment) == 0 {
		return fmt.Errorf("one of required FSC Client property is emtpy:ServiceName:%s, Region:%s, Environment:%s",
			c.ServiceName, c.Region, c.Environment)
	}

	metrics.Initialize(c.ServiceName, c.Region, string(c.Environment), c.Observer, c.Logger)

	featuresToCache := make(map[types.FeatureKey]types.FeatureParams)
	var featureKeys []types.FeatureKey
	for _, f := range features {
		featureKeys = append(featureKeys, f.FeatureKey)
		if f.CacheFeature {
			if f.FeatureJitter == 0 {
				f.FeatureJitter = defaultJitter
			}
			featuresToCache[f.FeatureKey] = f
		}
	}

	err := c.loadMetadata(ctx, featureKeys, manager)
	if err != nil {
		return fmt.Errorf("failed to load feature metadata: %w", err)
	}

	// Initialize caching
	if len(featuresToCache) > 0 {
		if c.Cache == nil {
			var clock *boundedCache.CachedClock
			if c.CacheConfig.CustomClock == nil {
				c.cachedclock = boundedCache.NewCachedClock()
				clock = c.cachedclock
				c.CacheConfig.CustomClock = clock
			}
			newCache, err := cache.NewTwitchBoundedCache(c.CacheConfig)
			if err != nil {
				return fmt.Errorf("failed to create boundedCache, %w", err)
			}
			c.Cache = newCache
		}
	}

	// Initialize DDB backends - for each account+table, initialize a DDB session

	// Key is FeatureKey
	featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
	// Key is - make sure each feature source is initialized once
	// TODO unify these two maps
	sources := make(map[string]source.IdentifiableFeatureSource)
	identifiableSource := make(map[string]source.IdentifiableFeatureSource)

	for key, feature := range c.metadata {
		backend := feature.GetDynamoDBBetaSource()
		if c.Environment == Prod {
			backend = feature.GetDynamoDBProdSource()
		}

		if val, ok := sources[backend.GetIdentifier()]; ok {
			featureSourceMap[key] = val
		} else {
			// TODO too many dependencies?
			ddb, err := source.CreateDynamoDBOnlineSource(c.Region, backend, feature.GetFeatureKey().Namespace,
				c.metadata, manager, c.CircuitConfig, c.HttpClient, c.ClientIdentifier, c.Logger)
			if err != nil {
				return fmt.Errorf("failed to initiate Dynamodb session for %+v: %w", backend, err)
			}
			featureSourceMap[key] = ddb
			sources[backend.GetIdentifier()] = ddb
			identifiableSource[ddb.GetIdentifier()] = ddb
		}
	}

	c.featureSources = featureSourceMap
	c.identifiableSource = identifiableSource
	c.featuresToCache = featuresToCache
	return nil
}

// Initialize will be deprecated, please use InitializeFeatures instead
func (c *Client) Initialize(ctx context.Context, features ...types.FeatureKey) error {
	var featureParams []types.FeatureParams
	for _, f := range features {
		featureParams = append(featureParams, types.FeatureParams{
			FeatureKey: f,
		})
	}
	return c.InitializeFeatures(ctx, featureParams...)
}

func (c *Client) GetDefaultFeatureSourceBuilder() source.FeatureSourceBuilder {
	return source.NewFeatureSourceBuilder(c.featureSources, c.identifiableSource, c.Cache, c.featuresToCache)
}

func (c *Client) GetFeatureMetadata(key types.FeatureKey) (metadata.Provider, error) {
	if m, ok := c.metadata[key]; ok {
		return m, nil
	}
	return nil, fmt.Errorf("could not find metadata for feature key: %v+", key)
}

// Stop function will stop the background clock
// the background clock is used to get the current time when setting and comparing ttl of the items in the cache
func (c *Client) Stop() {
	if c.cachedclock != nil {
		c.cachedclock.Stop()
	}
}
