package metadata

import (
	"errors"

	"github.com/aws/aws-sdk-go/aws"

	"code.justin.tv/amzn/TwitchFeatureGo"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/validations"
)

type floatAllowedRangeValidationMetadata struct {
	IdentifierGetterImpl
	validations.FloatAllowedRangeValidation
}

type intAllowedRangeValidationMetadata struct {
	IdentifierGetterImpl
	validations.IntAllowedRangeValidation
}

func AllowedRangeValidationServicer(id types.FeatureKey, validation *TwitchFeatureGo.Validation, valueType *TwitchFeatureGo.ValueType) (MetadataServicer, error) {
	if valueType == nil {
		return nil, errors.New("value type cannot be nil for validation")
	}

	// not yet supported
	if valueType.Shape != TwitchFeatureGo.ValueType_DATA_SHAPE_SCALAR {
		return &noOpValidationMetadata{IdentifierGetterImpl{id}, validations.UnsupportedValidation{}}, nil
	}

	rule := validation.GetAllowedRange()
	if rule == nil {
		return nil, errors.New("Expected allowed values to exist but is nil")
	}

	if rule.Max == nil && rule.Min == nil {
		return nil, errors.New("Expected a range, but neither min nor max is set")
	}

	switch valueType.Type {
	case TwitchFeatureGo.ValueType_DATA_TYPE_INTEGER:
		var maxVal, minVal *int64

		if max := rule.GetMax(); max != nil {
			maxVal = aws.Int64(max.GetIntegerVal())
		}
		if min := rule.GetMin(); min != nil {
			minVal = aws.Int64(min.GetIntegerVal())
		}

		return &intAllowedRangeValidationMetadata{
			IdentifierGetterImpl{id},
			validations.IntAllowedRangeValidation{
				Min:          minVal,
				Max:          maxVal,
				MaxInclusive: rule.MaxInclusive,
				MinInclusive: rule.MinInclusive,
				Replacer:     validations.NewReplacer(validation, valueType),
			},
		}, nil
	case TwitchFeatureGo.ValueType_DATA_TYPE_FLOAT:

		var maxVal, minVal *float64

		if max := rule.GetMax(); max != nil {
			maxVal = aws.Float64(max.GetFloatVal())
		}
		if min := rule.GetMin(); min != nil {
			minVal = aws.Float64(min.GetFloatVal())
		}

		return &floatAllowedRangeValidationMetadata{
			IdentifierGetterImpl{id},
			validations.FloatAllowedRangeValidation{
				Min:          minVal,
				Max:          maxVal,
				MaxInclusive: rule.MaxInclusive,
				MinInclusive: rule.MinInclusive,
				Replacer:     validations.NewReplacer(validation, valueType),
			},
		}, nil
	}

	// not yet supported data type
	return &noOpValidationMetadata{IdentifierGetterImpl{id}, validations.UnsupportedValidation{}}, nil
}
