package metadata

import (
	"errors"

	mapset "github.com/deckarep/golang-set"

	"code.justin.tv/amzn/TwitchFeatureGo"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/validations"
)

type intAllowedValuesValidationMetadata struct {
	IdentifierGetterImpl
	validations.IntAllowedValuesValidation
}

type floatAllowedValuesValidationMetadata struct {
	IdentifierGetterImpl
	validations.FloatAllowedValuesValidation
}

type stringAllowedValuesValidationMetadata struct {
	IdentifierGetterImpl
	validations.StringAllowedValuesValidation
}

func AllowedValueValidationServicer(id types.FeatureKey, validation *TwitchFeatureGo.Validation, valueType *TwitchFeatureGo.ValueType) (MetadataServicer, error) {
	if valueType == nil {
		return nil, errors.New("value type cannot be nil for validation")
	}

	// not yet supported
	if valueType.Shape != TwitchFeatureGo.ValueType_DATA_SHAPE_SCALAR {
		return &noOpValidationMetadata{IdentifierGetterImpl{id}, validations.UnsupportedValidation{}}, nil
	}

	rule := validation.GetAllowedValue()
	if rule == nil {
		return nil, errors.New("Expected allowed values to exist but is nil")
	}

	if len(rule.Set) == 0 {
		return nil, errors.New("Expected allowed values, but none set")
	}

	switch valueType.Type {
	case TwitchFeatureGo.ValueType_DATA_TYPE_INTEGER:
		values := make([]interface{}, len(rule.Set))
		for i, v := range rule.Set {
			values[i] = v.GetIntegerVal()
		}
		return &intAllowedValuesValidationMetadata{
			IdentifierGetterImpl{id},
			validations.IntAllowedValuesValidation{
				Values:   mapset.NewSetFromSlice(values),
				Replacer: validations.NewReplacer(validation, valueType),
			},
		}, nil
	case TwitchFeatureGo.ValueType_DATA_TYPE_FLOAT:
		values := make([]float64, len(rule.Set))
		for i, v := range rule.Set {
			values[i] = v.GetFloatVal()
		}
		return &floatAllowedValuesValidationMetadata{
			IdentifierGetterImpl{id},
			validations.FloatAllowedValuesValidation{
				Values:   values,
				Replacer: validations.NewReplacer(validation, valueType),
			},
		}, nil
	case TwitchFeatureGo.ValueType_DATA_TYPE_STRING:
		values := make([]interface{}, len(rule.Set))
		for i, v := range rule.Set {
			values[i] = v.GetStringVal()
		}
		return &stringAllowedValuesValidationMetadata{
			IdentifierGetterImpl{id},
			validations.StringAllowedValuesValidation{
				Values:   mapset.NewSetFromSlice(values),
				Replacer: validations.NewReplacer(validation, valueType),
			},
		}, nil
	}

	// not yet supported data type
	return &noOpValidationMetadata{IdentifierGetterImpl{id}, validations.UnsupportedValidation{}}, nil
}
