package metadata

import (
	"context"
	"fmt"

	"code.justin.tv/amzn/TwitchFeatureGo"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/store"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

func LoadMetadata(ctx context.Context, featureMetadataStore store.MetadataStore, ids []types.FeatureKey) (map[types.FeatureKey]Provider, error) {
	providers := make(map[types.FeatureKey]Provider)

	for _, id := range ids {
		rawMetadata, err := featureMetadataStore.Get(ctx, id.GetPath())
		if err != nil {
			return nil, fmt.Errorf("fetching metadata from store failed: %v", err)
		}
		servicer, err := toMetadataServicer(id, rawMetadata)
		if err != nil {
			return nil, fmt.Errorf("failed to interpret metadata: %v", err)
		}
		p, err := NewMetadataProvider(servicer, rawMetadata)
		if err != nil {
			return nil, err
		}
		providers[id] = p
	}

	return providers, nil
}

func toMetadataServicer(id types.FeatureKey, rawMetadata *TwitchFeatureGo.FeatureMetadata) (MetadataServicer, error) {
	validation := rawMetadata.Validation
	if validation == nil {
		return NewNoOpValidationServicer(id), nil
	}

	if rawMetadata.ValueType == nil {
		return NewNoOpValidationServicer(id), nil
	}

	if rule := validation.GetAllowedValue(); rule != nil {
		return AllowedValueValidationServicer(id, validation, rawMetadata.ValueType)
	}

	if rule := validation.GetAllowedRange(); rule != nil {
		return AllowedRangeValidationServicer(id, validation, rawMetadata.ValueType)
	}

	if rule := validation.GetAllValues(); rule != nil {
		return AllValuesValidationServicer(id, validation, rawMetadata.ValueType)
	}

	// not supported validation
	return NewNoOpValidationServicer(id), nil
}
