package metadata

import (
	"fmt"

	schema "code.justin.tv/amzn/TwitchFeatureGo"

	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/validations"
)

type IdentifierGetter interface {
	GetFeatureKey() *types.FeatureKey
}

type IdentifierGetterImpl struct {
	id types.FeatureKey
}

var _ IdentifierGetter = &IdentifierGetterImpl{}

func (i IdentifierGetterImpl) GetFeatureKey() *types.FeatureKey {
	return &i.id
}

type MetadataServicer interface {
	IdentifierGetter
	validations.Validater
}

//go:generate counterfeiter -o ../fakes/metadatafake/dynamodb_source.go . DynamoDBSource
type DynamoDBSource interface {
	GetIdentifier() string
	GetAccountId() string
	GetTable() string
	GetReadRole() string
	GetWriteRole() string
}

type dynamoDBSourceBackend struct {
	schema.DynamodbSource
}

var _ DynamoDBSource = &dynamoDBSourceBackend{}

func (d *dynamoDBSourceBackend) GetIdentifier() string {
	return fmt.Sprintf("%s:%s", d.AccountId, d.Table)
}

// TODO remove this default role - this should be filled in FR
func (d *dynamoDBSourceBackend) GetReadRole() string {
	if d.DynamodbSource.GetReadRole() == "" {
		return fmt.Sprintf("arn:aws:iam::%s:role/read_%s", d.GetAccountId(), d.GetTable())
	}
	return d.DynamodbSource.GetReadRole()
}

// TODO remove this default role - this should be filled in FR
func (d *dynamoDBSourceBackend) GetWriteRole() string {
	if d.DynamodbSource.GetWriteRole() == "" {
		return fmt.Sprintf("arn:aws:iam::%s:role/write_%s", d.GetAccountId(), d.GetTable())
	}
	return d.DynamodbSource.GetWriteRole()
}

type RequestSource interface {
	GetFeatureContextKey() string
}

type requestSourceBackend struct {
	schema.RequestSource
}

var _ RequestSource = &requestSourceBackend{}

type metadataImpl struct {
	MetadataServicer
	entity            []types.EntityType
	ddbBetaSource     DynamoDBSource
	ddbProdSource     DynamoDBSource
	requestBetaSource *requestSourceBackend
	requestProdSource *requestSourceBackend
	valueDataType     types.ValueDataType
	valueDataShape    types.ValueDataShape
}

var _ Provider = &metadataImpl{}

func NewMetadataProvider(servicer MetadataServicer, raw *schema.FeatureMetadata) (*metadataImpl, error) {
	m := &metadataImpl{MetadataServicer: servicer}

	if err := m.setEntity(raw); err != nil {
		return nil, err
	}
	if err := m.setDataValueType(raw); err != nil {
		return nil, err
	}
	if err := m.setDataValueShape(raw); err != nil {
		return nil, err
	}

	if raw.Source.BetaOnline.GetDynamodb() != nil {
		m.ddbBetaSource = &dynamoDBSourceBackend{
			DynamodbSource: *raw.Source.BetaOnline.GetDynamodb(),
		}
	}
	if raw.Source.ProdOnline.GetDynamodb() != nil {
		m.ddbProdSource = &dynamoDBSourceBackend{
			DynamodbSource: *raw.Source.ProdOnline.GetDynamodb(),
		}
	}

	return m, nil
}

func (m *metadataImpl) setEntity(raw *schema.FeatureMetadata) error {
	for _, i := range raw.Entity {
		switch i {
		case schema.Entity_ENTITY_OTHER:
			m.entity = append(m.entity, types.OTHER)
		case schema.Entity_ENTITY_CHANNEL:
			m.entity = append(m.entity, types.CHANNEL)
		case schema.Entity_ENTITY_DEVICE:
			m.entity = append(m.entity, types.DEVICE)
		case schema.Entity_ENTITY_QUERY:
			m.entity = append(m.entity, types.QUERY)
		case schema.Entity_ENTITY_USER:
			m.entity = append(m.entity, types.USER)
		case schema.Entity_ENTITY_CATEGORY:
			m.entity = append(m.entity, types.CATEGORY)
		default:
			return fmt.Errorf("unknown entity type:%s", i)
		}
	}
	return nil
}

func (m *metadataImpl) setDataValueType(raw *schema.FeatureMetadata) error {
	switch raw.ValueType.Type {
	case schema.ValueType_DATA_TYPE_STRING:
		m.valueDataType = types.STRING
	case schema.ValueType_DATA_TYPE_INTEGER:
		m.valueDataType = types.INTEGER
	case schema.ValueType_DATA_TYPE_FLOAT:
		m.valueDataType = types.FLOAT
	default:
		return fmt.Errorf("unkown data value type:%s", raw.ValueType.Type)
	}
	return nil
}

func (m *metadataImpl) setDataValueShape(raw *schema.FeatureMetadata) error {
	switch raw.ValueType.Shape {
	case schema.ValueType_DATA_SHAPE_SCALAR:
		m.valueDataShape = types.SCALAR
	case schema.ValueType_DATA_SHAPE_VECTOR:
		m.valueDataShape = types.VECTOR
	case schema.ValueType_DATA_SHAPE_LIST:
		m.valueDataShape = types.LIST
	case schema.ValueType_DATA_SHAPE_BLOB:
		m.valueDataShape = types.BLOB
	default:
		return fmt.Errorf("unkown data value shape:%s", raw.ValueType.Shape)
	}
	return nil
}

func (m *metadataImpl) GetEntity() []types.EntityType {
	return m.entity
}

func (m *metadataImpl) GetDynamoDBBetaSource() DynamoDBSource {
	return m.ddbBetaSource
}

func (m *metadataImpl) GetDynamoDBProdSource() DynamoDBSource {
	return m.ddbProdSource
}

func (m *metadataImpl) GetRequestBetaSource() RequestSource {
	return m.requestBetaSource
}

func (m *metadataImpl) GetRequestProdSource() RequestSource {
	return m.requestProdSource
}

func (m *metadataImpl) GetValueDataType() types.ValueDataType {
	return m.valueDataType
}

func (m *metadataImpl) GetValueDataShape() types.ValueDataShape {
	return m.valueDataShape
}

//go:generate counterfeiter -o ../fakes/metadatafake/provider.go . Provider
// Provide access to all fields of feature metadata
type Provider interface {
	MetadataServicer
	GetEntity() []types.EntityType
	GetDynamoDBBetaSource() DynamoDBSource
	GetDynamoDBProdSource() DynamoDBSource
	GetRequestBetaSource() RequestSource
	GetRequestProdSource() RequestSource
	GetValueDataType() types.ValueDataType
	GetValueDataShape() types.ValueDataShape
}
