package metadata

import (
	"context"
	"testing"

	tf "code.justin.tv/amzn/TwitchFeatureGo"

	store "code.justin.tv/amzn/TwitchFeatureStoreClient/fakes/store"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/validations"
)

func TestGenerateMetadata(t *testing.T) {
	// pass multiple ids, get multiple servicers
	ids := []types.FeatureKey{
		{Namespace: "vx",
			Version:   1,
			FeatureID: "no_followers"},
		{Namespace: "rec",
			Version:   2,
			FeatureID: "viewing_mins"},
	}

	fakeStore := &store.FakeMetadataStore{}

	fakeStore.GetReturnsOnCall(0, &tf.FeatureMetadata{
		ValueType: &tf.ValueType{
			Type:  tf.ValueType_DATA_TYPE_INTEGER,
			Shape: tf.ValueType_DATA_SHAPE_SCALAR,
		},
		Validation: &tf.Validation{
			Rule: &tf.Validation_AllowedRange{
				AllowedRange: &tf.RangeValidation{
					Min:          &tf.Value{Val: &tf.Value_FloatVal{FloatVal: 0}},
					MinInclusive: true,
				}}},
		Source: &tf.Source{
			BetaOnline: &tf.OnlineSource{
				Source: &tf.OnlineSource_Dynamodb{
					Dynamodb: &tf.DynamodbSource{
						AccountId: "123",
						Table:     "test_table",
						Role:      "read_test_table",
					},
				},
			},
			ProdOnline: &tf.OnlineSource{
				Source: &tf.OnlineSource_Dynamodb{
					Dynamodb: &tf.DynamodbSource{
						AccountId: "321",
						Table:     "test_table",
						Role:      "read_test_table",
					},
				},
			},
		},
	}, nil)
	fakeStore.GetReturnsOnCall(1, &tf.FeatureMetadata{
		ValueType: &tf.ValueType{
			Type:  tf.ValueType_DATA_TYPE_INTEGER,
			Shape: tf.ValueType_DATA_SHAPE_SCALAR,
		},
		Source: &tf.Source{
			BetaOnline: &tf.OnlineSource{
				Source: &tf.OnlineSource_Dynamodb{
					Dynamodb: &tf.DynamodbSource{
						AccountId: "123",
						Table:     "test_table",
						Role:      "read_test_table",
					},
				},
			},
			ProdOnline: &tf.OnlineSource{
				Source: &tf.OnlineSource_Dynamodb{
					Dynamodb: &tf.DynamodbSource{
						AccountId: "321",
						Table:     "test_table",
						Role:      "read_test_table",
					},
				},
			},
		},
	}, nil)

	actual, err := LoadMetadata(context.Background(), fakeStore, ids)

	if err != nil {
		t.Errorf("Expected no error but got %v", err)
	}

	if len(actual) != 2 {
		t.Errorf("Unexpected metadata servicer count; expected %v but got %v", 2, len(actual))
	}

	servicer1, ok1 := actual[ids[0]]
	_, ok2 := actual[ids[1]]
	if !ok1 || !ok2 {
		t.Error("Expected feature keys not in the metadata")
	}

	feat := servicer1.GetFeatureKey()
	if feat.Namespace != "vx" || feat.FeatureID != "no_followers" || feat.Version != 1 {
		t.Errorf("Unexpected GetFeatureKey return value")
	}

	// sanity integration test
	if _, validationErr := validations.ValidateFeatureAndReplace(servicer1, &types.IntFeature{Val: -1}); validationErr != nil {
		t.Errorf("Unexpected vx/no_followers/1 servicer behavior")
	}
}
