package metadata

import (
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
	"reflect"
	"testing"

	"code.justin.tv/amzn/TwitchFeatureGo"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/validations"
)

type testCase struct {
	name              string
	input             inputArgs
	expectedError     bool
	validaterTestCase *validaterTest
}

type inputArgs struct {
	id          types.FeatureKey
	rawMetadata *TwitchFeatureGo.FeatureMetadata
}

type validaterTest struct {
	value         types.FeatureValue
	expected      types.FeatureValue
	expectedError bool
}

func newToMetadataInput(metadata *TwitchFeatureGo.FeatureMetadata) inputArgs {
	return inputArgs{id: types.FeatureKey{
		Namespace: "vx",
		FeatureID: "hello",
		Version:   1,
	}, rawMetadata: metadata}
}

// tests whether the metadata servicer works in general with a quick test
func runGenerateMetadataTests(t *testing.T, tests []testCase) {
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			servicer, err := toMetadataServicer(tt.input.id, tt.input.rawMetadata)
			if err != nil && !tt.expectedError {
				t.Errorf("%v: expected no error, but got error: %v", tt.name, err)
			} else if err == nil && tt.expectedError {
				t.Errorf("%v: expected error, but got no error", tt.name)
			}
			if err == nil {
				vt := tt.validaterTestCase
				actual, err := validations.ValidateFeatureAndReplace(servicer, vt.value)
				actualVal := actual.Value()
				expectedVal := vt.expected.Value()
				if err != nil {
					t.Errorf("actual value error: %v", err)
				}
				if actualVal != expectedVal {
					t.Errorf("%v: expected %v but got %v", tt.name, expectedVal, actualVal)
				}
			}
		})
	}
}

func newProtobufIntValue(val int64) *TwitchFeatureGo.Value {
	return &TwitchFeatureGo.Value{Val: &TwitchFeatureGo.Value_IntegerVal{IntegerVal: val}}
}
func newProtobufFloatValue(val float64) *TwitchFeatureGo.Value {
	return &TwitchFeatureGo.Value{Val: &TwitchFeatureGo.Value_FloatVal{FloatVal: val}}
}
func newProtobufStringValue(val string) *TwitchFeatureGo.Value {
	return &TwitchFeatureGo.Value{Val: &TwitchFeatureGo.Value_StringVal{StringVal: val}}
}

func newProtobufMetdata(validation *TwitchFeatureGo.Validation, kind reflect.Kind) *TwitchFeatureGo.FeatureMetadata {
	switch kind {
	case reflect.Int64:
		return &TwitchFeatureGo.FeatureMetadata{
			ValueType: &TwitchFeatureGo.ValueType{
				Type: TwitchFeatureGo.ValueType_DATA_TYPE_INTEGER, Shape: TwitchFeatureGo.ValueType_DATA_SHAPE_SCALAR},
			Validation: validation,
		}
	case reflect.Float64:
		return &TwitchFeatureGo.FeatureMetadata{
			ValueType: &TwitchFeatureGo.ValueType{
				Type: TwitchFeatureGo.ValueType_DATA_TYPE_FLOAT, Shape: TwitchFeatureGo.ValueType_DATA_SHAPE_SCALAR},
			Validation: validation,
		}
	case reflect.String:
		return &TwitchFeatureGo.FeatureMetadata{
			ValueType: &TwitchFeatureGo.ValueType{
				Type: TwitchFeatureGo.ValueType_DATA_TYPE_STRING, Shape: TwitchFeatureGo.ValueType_DATA_SHAPE_SCALAR},
			Validation: validation,
		}
	}
	return nil
}
