package metrics

import (
	"fmt"
	"log"
	"time"

	logging "code.justin.tv/amzn/TwitchLogging"
	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"

	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

type metadata struct {
	featureConsumer string
	region          string
	stage           string
	observer        telemetry.SampleObserver
	logger          logging.Logger
}

var reporterMetadata = &metadata{}

const (
	DimensionFeature = "Feature"

	MetricAttemptCount = "AttemptCount"
	MetricLatency      = "Latency"
)

var reporterRollUpDimensions = [][]string{
	{DimensionFeature, telemetry.DimensionStage},
	{DimensionFeature, telemetry.DimensionService, telemetry.DimensionStage},
}

// Initialize is used to initialize essential metrics metadata and should be called once at FSC initialization
func Initialize(consumer, region, stage string, observer telemetry.SampleObserver, logger logging.Logger) {
	reporterMetadata.featureConsumer = consumer
	reporterMetadata.region = region
	reporterMetadata.stage = stage
	reporterMetadata.observer = observer
	reporterMetadata.logger = logger
}

type Reporter interface {
	ReportAttempt(value float64)
	ReportLatency(duration time.Duration)
}

var _ Reporter = &reporterImpl{}

type reporterImpl struct {
	builder  telemetry.SampleBuilder
	observer telemetry.SampleObserver
	logger   logging.Logger
}

func NewReporter(fk types.FeatureKey) Reporter {
	reporter := &reporterImpl{
		builder: telemetry.SampleBuilder{
			ProcessIdentifier: identifier.ProcessIdentifier{
				Service: reporterMetadata.featureConsumer,
				Region:  reporterMetadata.region,
				Stage:   reporterMetadata.stage,
			},
			Dimensions: map[string]string{
				DimensionFeature: fmt.Sprintf("%s/%s/%d", fk.Namespace, fk.FeatureID, fk.Version),
			},
		},
		observer: reporterMetadata.observer,
		logger:   reporterMetadata.logger,
	}
	return reporter
}

func (r *reporterImpl) ReportAttempt(value float64) {
	if r.observer == nil {
		return
	}
	r.reportSample(MetricAttemptCount, value)
}

// ReportSample builds a new sample using SampleBuilder.Build and sends it to the observer.
func (r *reporterImpl) reportSample(metricName string, value float64) {
	sample, err := r.builder.Build(metricName, value, telemetry.UnitCount)
	if err == nil {
		sample.RollupDimensions = reporterRollUpDimensions
		r.observer.ObserveSample(sample)
	} else {
		if r.logger != nil {
			r.logger.Log("Failed to report sample", "err", err.Error())
		} else {
			log.Printf("Failed to report sample: %v", err)
		}
	}
}

// ReportDurationSample builds a new sample using SampleBuilder.BuildDurationSample and sends it to the observer.
func (r *reporterImpl) ReportLatency(duration time.Duration) {
	if r.observer == nil {
		return
	}
	sample, err := r.builder.BuildDurationSample(MetricLatency, duration)
	if err == nil {
		sample.RollupDimensions = reporterRollUpDimensions
		r.observer.ObserveSample(sample)
	} else {
		if r.logger != nil {
			r.logger.Log("Failed to report duration sample", "err", err.Error())
		} else {
			log.Printf("Failed to report duration sample: %v", err)
		}
	}
}
