package source

import (
	"context"
	"errors"
	"fmt"
	"time"

	"golang.org/x/sync/errgroup"

	"code.justin.tv/amzn/TwitchFeatureStoreClient/cache"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/jitter"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/metrics"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

type Failure string

// InvalidFeature is used to indicate features that fail the validation rules
// defined in the corresponding feature metadata.
type InvalidFeature struct {
	types.InstanceKey
	Value   types.FeatureValue
	Failure Failure
}

// FeatureSource and FeatureSourceBulkAccess are two interface for
// client service to implement when it needs to access features that
// are not provided by the Feature Store. Some examples are features
// coming from request or some other live service like Liveline.

// FeatureSource is an interface for access single feature instance.
type FeatureSource interface {
	Get(ctx context.Context, feature *types.FeatureInstance) error
}

// FeatureSourceBulkAccess is an interface for access multiple feature instances.
// They can be from the same feature_id or different feature_ids.
type FeatureSourceBulkAccess interface {
	BulkGet(ctx context.Context, features []*types.FeatureInstance) error
}

// FeatureSourceBuilder is used to assemble feature sources used to fetch different features.
// The lifecycle of the FeatureSourceBuilder is within a request. All functions in this interface
// are not thread-safe
type FeatureSourceBuilder interface {
	// Use includes any features that are stored and can be fetched in OFS
	// InstanceKey of unregistered FeatureKey will be ignored
	Use(instanceKeys ...types.InstanceKey) FeatureSourceBuilder

	// Add includes a feature which is provided by source.
	Add(source FeatureSource, instanceKey types.InstanceKey) FeatureSourceBuilder

	// Bundle includes a set of features which are provided by the same source.
	// All features specified through this function will be fetched in a single request to source.
	Bundle(source FeatureSourceBulkAccess, instanceKeys ...types.InstanceKey) FeatureSourceBuilder

	// Build initiates concurrent requests to the feature sources for features that have been added/bundled
	// in the builder. After all features have been fetched, it also performs validations on the feature values
	// (only feature with RequestSource) based on rules defined in their feature metadata. Finally, it returns
	// the results as FeatureGetter.
	Build(ctx context.Context) (types.FeatureGetter, []InvalidFeature, error)
}

type singleSource struct {
	source FeatureSource
	key    types.InstanceKey
}

type bulkSource struct {
	source FeatureSourceBulkAccess
	keys   []types.InstanceKey
}

type featureSourceBuilderImpl struct {
	singleSources []*singleSource
	batchSources  []*bulkSource

	reporters map[types.FeatureKey]metrics.Reporter

	featureAccessor types.InstanceAccessor

	// These are needed for default DDB feature source implementation
	featureSourceMap    map[types.FeatureKey]IdentifiableFeatureSource
	sourceIdFeatureMap  map[string][]types.InstanceKey // maps identifiable feature source to list of InstanceKeys
	identifiableSources map[string]IdentifiableFeatureSource

	// cache for feature values
	cache           cache.Cache
	featuresToCache map[types.FeatureKey]types.FeatureParams
	jitterGenerator *jitter.JitterGenerator
}

var _ FeatureSourceBuilder = &featureSourceBuilderImpl{}

func NewFeatureSourceBuilder(featureSourceMap map[types.FeatureKey]IdentifiableFeatureSource,
	sources map[string]IdentifiableFeatureSource, cache cache.Cache,
	featuresToCache map[types.FeatureKey]types.FeatureParams) *featureSourceBuilderImpl {
	return &featureSourceBuilderImpl{
		featureAccessor:     types.NewInstanceIndexer(),
		featureSourceMap:    featureSourceMap,
		sourceIdFeatureMap:  make(map[string][]types.InstanceKey),
		identifiableSources: sources,
		reporters:           make(map[types.FeatureKey]metrics.Reporter),
		cache:               cache,
		featuresToCache:     featuresToCache,
		jitterGenerator:     jitter.NewJitterGenerator(),
	}
}

func (f *featureSourceBuilderImpl) Use(instanceKeys ...types.InstanceKey) FeatureSourceBuilder {
	if len(instanceKeys) == 0 {
		return f
	}
	for _, i := range instanceKeys {
		if ifs, ok := f.featureSourceMap[i.FeatureKey]; ok {
			featureList := f.sourceIdFeatureMap[ifs.GetIdentifier()]
			featureList = append(featureList, i)
			f.sourceIdFeatureMap[ifs.GetIdentifier()] = featureList
		}

	}
	return f
}

func (f *featureSourceBuilderImpl) registerReporter(key types.FeatureKey) {
	if _, ok := f.reporters[key]; !ok {
		f.reporters[key] = metrics.NewReporter(key)
	}
}

func (f *featureSourceBuilderImpl) Add(source FeatureSource, instanceKey types.InstanceKey) FeatureSourceBuilder {
	f.featureAccessor.SetInstance(instanceKey, nil)
	f.singleSources = append(f.singleSources, &singleSource{
		source: source,
		key:    instanceKey,
	})
	f.registerReporter(instanceKey.FeatureKey)
	return f
}

// TODO Note we do not deduplicate source. If a single source is "bundled" multiple times, they will be called multiple times
// TODO a single OFS table should be treated as a FeatureSourceBulkAccess implementation?
func (f *featureSourceBuilderImpl) Bundle(source FeatureSourceBulkAccess, instanceKeys ...types.InstanceKey) FeatureSourceBuilder {
	if len(instanceKeys) == 0 {
		return f
	}
	f.batchSources = append(f.batchSources, &bulkSource{
		source: source,
		keys:   instanceKeys,
	})
	for _, i := range instanceKeys {
		f.featureAccessor.SetInstance(i, nil)
		f.registerReporter(i.FeatureKey)
	}
	return f
}

// TODO build should only be run once per object - use sync.Once
func (f *featureSourceBuilderImpl) Build(ctx context.Context) (types.FeatureGetter, []InvalidFeature, error) {
	// special handling for features stored in OFS
	if len(f.sourceIdFeatureMap) > 0 {
		for k, v := range f.sourceIdFeatureMap {
			if src, ok := f.identifiableSources[k]; ok {
				f.Bundle(src, v...)
			} else {
				return nil, nil, fmt.Errorf("unregistered feature source:%s", k)
			}
		}
	}

	singleGroup, ctxSingle := errgroup.WithContext(ctx)
	for _, v := range f.singleSources {
		func(s *singleSource) {
			singleGroup.Go(func() error {
				instance, err := f.featureAccessor.GetInstance(s.key)
				if err != nil {
					return err
				}
				key := instance.GetCacheKey()
				if f.cache != nil {
					if val, ok := f.cache.Get(key); ok {
						if featureVal, ok := val.(types.FeatureValue); ok {
							instance.Value = featureVal
							return nil
						}
					}
				}

				startTime := time.Now()
				err = s.source.Get(ctxSingle, instance)
				if r, ok := f.reporters[s.key.FeatureKey]; ok {
					r.ReportAttempt(1)
					if err == nil {
						// only report latency for succeeded requests
						r.ReportLatency(time.Since(startTime))
					}
				}
				if err != nil {
					return err
				}
				if f.cache != nil {
					f.addInstancesToCache(instance)
				}
				return nil
			})
		}(v)
	}

	bulkGroup, ctxBulk := errgroup.WithContext(ctx)
	for _, v := range f.batchSources {
		func(b *bulkSource) {
			bulkGroup.Go(func() error {
				var instances []*types.FeatureInstance
				for _, key := range b.keys {
					instance, err := f.featureAccessor.GetInstance(key)
					if err != nil {
						return err
					}
					cacheKey := instance.GetCacheKey()
					if f.cache != nil {
						if val, ok := f.cache.Get(cacheKey); ok {
							if featureVal, ok := val.(types.FeatureValue); ok {
								instance.Value = featureVal
								continue
							}
						}
					}
					instances = append(instances, instance)
				}
				if len(instances) == 0 {
					return nil
				}
				startTime := time.Now()
				err := b.source.BulkGet(ctxBulk, instances)
				totalTime := time.Since(startTime)
				// TODO currently only number of attempts is reported. Do we care about success / failures?
				for _, i := range instances {
					if r, ok := f.reporters[i.FeatureKey]; ok {
						r.ReportAttempt(1)
						if err == nil {
							r.ReportLatency(totalTime)
						}
					}
				}
				if err != nil {
					return err
				}
				if f.cache != nil {
					f.addInstancesToCache(instances...)
				}
				return nil
			})
		}(v)
	}

	// TODO single group and bulk group can be in parallel as well
	singleErr := singleGroup.Wait()
	bulkErr := bulkGroup.Wait()

	if singleErr != nil || bulkErr != nil {
		err := errors.New("failed to fetch features from feature sources")
		if singleErr != nil {
			err = fmt.Errorf("failed to fetch features from FeatureSource: %w", singleErr)
		}
		if bulkErr != nil {
			err = fmt.Errorf("failed to fetch features from FeatureSourceBulkAccess: %w", bulkErr)
		}
		return nil, nil, err
	}

	// TODO validation plug-in
	return f.featureAccessor, nil, nil
}

func (f *featureSourceBuilderImpl) addInstancesToCache(instances ...*types.FeatureInstance) {
	for _, i := range instances {
		cacheKey := i.GetCacheKey()
		if params, ok := f.featuresToCache[i.FeatureKey]; ok {
			var jitterDur time.Duration
			var ttl time.Duration
			if params.CacheTTL > 0 {
				ttl = params.CacheTTL
				if params.FeatureJitter > 0 {
					jitterDur = params.FeatureJitter
				}
				jitterDur = f.jitterGenerator.Jitter(jitterDur)
			}
			f.cache.SetWithExpiration(cacheKey, i.Value, ttl+jitterDur)
		}
	}
}
