package source_test

import (
	"context"
	"errors"
	"reflect"
	"testing"
	"time"

	cache "code.justin.tv/amzn/TwitchBoundedCache"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/source"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

type testKeys struct {
	key1, key2, key3, key4 types.InstanceKey
}

func newTestKeys() testKeys {
	bt := testKeys{
		key1: types.InstanceKey{
			FeatureKey: types.FeatureKey{
				FeatureID: "test_id_1",
				Namespace: "test",
				Version:   0,
			},
			Entities: []types.Entity{
				{
					Type: types.CATEGORY,
					Id:   "test_category_1",
				},
				{
					Type: types.USER,
					Id:   "test_user_1",
				},
			},
		},
		key2: types.InstanceKey{
			FeatureKey: types.FeatureKey{
				FeatureID: "test_id_2",
				Namespace: "test",
				Version:   1,
			},
			Entities: []types.Entity{
				{
					Type: types.QUERY,
					Id:   "test_query_2",
				},
			},
		},
		key3: types.InstanceKey{
			FeatureKey: types.FeatureKey{
				FeatureID: "test_id_3",
				Namespace: "test",
				Version:   0,
			},
			Entities: []types.Entity{
				{
					Type: types.OTHER,
					Id:   "test_other_3",
				},
			},
		},
		key4: types.InstanceKey{
			FeatureKey: types.FeatureKey{
				FeatureID: "test_id_3",
				Namespace: "test",
				Version:   0,
			},
			Entities: []types.Entity{
				{
					Type: types.OTHER,
					Id:   "test_other_4",
				},
			},
		},
	}

	return bt
}

// Implementations of FeatureSource and FeatureSourceBulkAccess to facilitate tests
type fakeIdSource struct {
	name   string
	isErr  bool
	expect func(key types.InstanceKey) types.FeatureValue
}

func (t *fakeIdSource) BulkGet(ctx context.Context, features []*types.FeatureInstance) error {
	if t.isErr {
		return errors.New("test error")
	}
	for _, f := range features {
		f.Value = t.expect(f.InstanceKey)
	}
	return nil
}

func (t *fakeIdSource) Get(ctx context.Context, feature *types.FeatureInstance) error {
	if t.isErr {
		return errors.New("test error")
	}
	feature.Value = t.expect(feature.InstanceKey)
	return nil
}

func (t *fakeIdSource) GetIdentifier() string {
	return t.name
}

func Test_FeatureSourceBuilder_Use_Build(t *testing.T) {
	keys := newTestKeys()
	tests := []struct {
		name     string
		builder  source.FeatureSourceBuilder
		testKeys []types.InstanceKey
		expected []types.FeatureInstance
		buildErr bool
		getErr   bool
	}{
		{
			name: "success - single feature",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				featureSourceMap[keys.key1.FeatureKey] = s

				sources := make(map[string]source.IdentifiableFeatureSource)
				sources[s.GetIdentifier()] = s

				return source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)
			}(),
			testKeys: []types.InstanceKey{keys.key1},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
		},
		{
			name: "success - multiple keys multiple sources",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				s2 := &fakeIdSource{
					name:  "source2",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key2) {
							return &types.IntFeature{Val: 2}
						}
						return nil
					},
				}
				featureSourceMap[keys.key1.FeatureKey] = s1
				featureSourceMap[keys.key2.FeatureKey] = s2

				sources := make(map[string]source.IdentifiableFeatureSource)
				sources[s1.GetIdentifier()] = s1
				sources[s2.GetIdentifier()] = s2

				return source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)
			}(),
			testKeys: []types.InstanceKey{keys.key1, keys.key2},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key2,
					Value:       &types.IntFeature{Val: 2},
				},
			},
		},
		{
			name: "success - multiple keys single source",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						if reflect.DeepEqual(key, keys.key3) {
							return &types.IntFeature{Val: 3}
						}
						return nil
					},
				}
				featureSourceMap[keys.key1.FeatureKey] = s
				featureSourceMap[keys.key3.FeatureKey] = s

				sources := make(map[string]source.IdentifiableFeatureSource)
				sources[s.GetIdentifier()] = s

				return source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)
			}(),
			testKeys: []types.InstanceKey{keys.key1, keys.key3},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key3,
					Value:       &types.IntFeature{Val: 3},
				},
			},
		},
		{
			name: "fail - missing source for given key",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				featureSourceMap[keys.key1.FeatureKey] = s

				sources := make(map[string]source.IdentifiableFeatureSource)

				return source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)
			}(),
			testKeys: []types.InstanceKey{keys.key1},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
			buildErr: true,
		},
		{
			name: "fail - feature source fail",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s := &fakeIdSource{
					name:  "source1",
					isErr: true,
				}
				featureSourceMap[keys.key1.FeatureKey] = s

				sources := make(map[string]source.IdentifiableFeatureSource)
				sources[s.GetIdentifier()] = s
				return source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)
			}(),
			testKeys: []types.InstanceKey{keys.key1},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
			buildErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			tt.builder.Use(tt.testKeys...)
			getter, _, err := tt.builder.Build(context.Background())
			if (err != nil) != tt.buildErr {
				t.Errorf("Build() error = %v, wantErr %v", err, tt.buildErr)
			}

			if err == nil {
				for _, fi := range tt.expected {
					val, err := getter.Get(fi.InstanceKey)
					if (err != nil) != tt.getErr {
						t.Errorf("FeatureGetter.Get() error = %v, wantErr %v", err, tt.getErr)
					}
					if !reflect.DeepEqual(val.Value(), fi.Value.Value()) {
						t.Errorf("Get() got = %v, want %v", val.Value(), fi.Value.Value())
					}
				}
			}
		})
	}
}

func Test_FeatureSourceBuilder_Add_Build(t *testing.T) {
	keys := newTestKeys()
	tests := []struct {
		name     string
		builder  source.FeatureSourceBuilder
		expected []types.FeatureInstance
		buildErr bool
		getErr   bool
	}{
		{
			name: "success - single source",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				b.Add(s, keys.key1)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
		},
		{
			name: "success - multiple sources",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				s2 := &fakeIdSource{
					name:  "source2",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key2) {
							return &types.IntFeature{Val: 2}
						}
						return nil
					},
				}

				b.Add(s1, keys.key1)
				b.Add(s2, keys.key2)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key2,
					Value:       &types.IntFeature{Val: 2},
				},
			},
		},
		{
			name: "success - same sources multiple keys",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						if reflect.DeepEqual(key, keys.key3) {
							return &types.IntFeature{Val: 3}
						}
						return nil
					},
				}

				b.Add(s1, keys.key1)
				b.Add(s1, keys.key3)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key3,
					Value:       &types.IntFeature{Val: 3},
				},
			},
		},
		{
			name: "fail - missing key",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						if reflect.DeepEqual(key, keys.key3) {
							return &types.IntFeature{Val: 3}
						}
						return nil
					},
				}

				b.Add(s1, keys.key1)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key3,
					Value:       &types.IntFeature{Val: 3},
				},
			},
			getErr: true,
		},
		{
			name: "fail - source failure",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: true,
				}

				b.Add(s1, keys.key1)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
			buildErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			getter, _, err := tt.builder.Build(context.Background())
			if (err != nil) != tt.buildErr {
				t.Errorf("Build() error = %v, wantErr %v", err, tt.buildErr)
			}

			if err == nil {
				for _, fi := range tt.expected {
					val, err := getter.Get(fi.InstanceKey)
					if (err != nil) != tt.getErr {
						t.Errorf("FeatureGetter.Get() error = %v, wantErr %v", err, tt.getErr)
					}
					if err == nil && !reflect.DeepEqual(val.Value(), fi.Value.Value()) {
						t.Errorf("Get() got = %v, want %v", val, fi.Value)
					}
				}
			}
		})
	}
}

func Test_FeatureSourceBuilder_Bundle_Build(t *testing.T) {
	keys := newTestKeys()
	tests := []struct {
		name     string
		builder  source.FeatureSourceBuilder
		expected []types.FeatureInstance
		buildErr bool
		getErr   bool
	}{
		{
			name: "success - single source multiple keys",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						if reflect.DeepEqual(key, keys.key2) {
							return &types.IntFeature{Val: 2}
						}
						return nil
					},
				}
				b.Bundle(s1, keys.key1, keys.key2)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key2,
					Value:       &types.IntFeature{Val: 2},
				},
			},
		},
		{
			name: "success - single source single key",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				b.Bundle(s1, keys.key1)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
		},
		{
			name: "success - multiple sources multiple keys",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				s2 := &fakeIdSource{
					name:  "source2",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key2) {
							return &types.IntFeature{Val: 2}
						}
						if reflect.DeepEqual(key, keys.key3) {
							return &types.IntFeature{Val: 3}
						}
						return nil
					},
				}
				s3 := &fakeIdSource{
					name:  "source3",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key4) {
							return &types.IntFeature{Val: 4}
						}
						return nil
					},
				}
				b.Bundle(s1, keys.key1)
				b.Bundle(s2, keys.key2, keys.key3)
				b.Add(s3, keys.key4)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key2,
					Value:       &types.IntFeature{Val: 2},
				},
				{
					InstanceKey: keys.key3,
					Value:       &types.IntFeature{Val: 3},
				},
				{
					InstanceKey: keys.key4,
					Value:       &types.IntFeature{Val: 4},
				},
			},
		},
		{
			name: "fail - missing key",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				b.Bundle(s1, keys.key1)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key2,
					Value:       &types.IntFeature{Val: 2},
				},
			},
			getErr: true,
		},
		{
			name: "fail - source failure",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				sources := make(map[string]source.IdentifiableFeatureSource)
				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, nil, nil)

				s1 := &fakeIdSource{
					name:  "source1",
					isErr: true,
				}
				b.Bundle(s1, keys.key1)
				return b
			}(),
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
			buildErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			getter, _, err := tt.builder.Build(context.Background())
			if (err != nil) != tt.buildErr {
				t.Errorf("Build() error = %v, wantErr %v", err, tt.buildErr)
			}

			if err == nil {
				for _, fi := range tt.expected {
					val, err := getter.Get(fi.InstanceKey)
					if (err != nil) != tt.getErr {
						t.Errorf("FeatureGetter.Get() error = %v, wantErr %v", err, tt.getErr)
					}
					if err == nil && !reflect.DeepEqual(val.Value(), fi.Value.Value()) {
						t.Errorf("Get() got = %v, want %v", val, fi.Value)
					}
				}
			}
		})
	}
}

func Test_FeatureSourceBuilder_Cache_Build(t *testing.T) {
	keys := newTestKeys()
	tests := []struct {
		name     string
		builder  source.FeatureSourceBuilder
		testKeys []types.InstanceKey
		expected []types.FeatureInstance
		buildErr bool
		getErr   bool
	}{
		{
			name: "success - single source",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				featureSourceMap[keys.key1.FeatureKey] = s

				sources := make(map[string]source.IdentifiableFeatureSource)
				sources[s.GetIdentifier()] = s

				fCache, _ := cache.NewCache(cache.DefaultConfig)
				featuresToCache := map[types.FeatureKey]types.FeatureParams{
					keys.key1.FeatureKey: {
						FeatureKey:    keys.key1.FeatureKey,
						CacheTTL:      1 * time.Minute,
						FeatureJitter: time.Second,
					},
				}

				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, fCache, featuresToCache)
				b.Add(s, keys.key1)
				return b
			}(),
			testKeys: []types.InstanceKey{keys.key1},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
			},
		},
		{
			name: "success - bulksource",
			builder: func() source.FeatureSourceBuilder {
				featureSourceMap := make(map[types.FeatureKey]source.IdentifiableFeatureSource)
				s1 := &fakeIdSource{
					name:  "source1",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key1) {
							return &types.IntFeature{Val: 1}
						}
						return nil
					},
				}
				s2 := &fakeIdSource{
					name:  "source2",
					isErr: false,
					expect: func(key types.InstanceKey) types.FeatureValue {
						if reflect.DeepEqual(key, keys.key2) {
							return &types.IntFeature{Val: 2}
						}
						return nil
					},
				}
				featureSourceMap[keys.key1.FeatureKey] = s1
				featureSourceMap[keys.key2.FeatureKey] = s2

				sources := make(map[string]source.IdentifiableFeatureSource)
				sources[s1.GetIdentifier()] = s1
				sources[s2.GetIdentifier()] = s2

				fCache, _ := cache.NewCache(cache.DefaultConfig)
				featuresToCache := map[types.FeatureKey]types.FeatureParams{
					keys.key1.FeatureKey: {
						FeatureKey:    keys.key1.FeatureKey,
						CacheTTL:      1 * time.Minute,
						FeatureJitter: time.Second,
					},
					keys.key2.FeatureKey: {
						FeatureKey:    keys.key2.FeatureKey,
						CacheTTL:      1 * time.Minute,
						FeatureJitter: time.Second,
					},
				}

				b := source.NewFeatureSourceBuilder(featureSourceMap, sources, fCache, featuresToCache)
				b.Use(keys.key1, keys.key2)
				return b
			}(),
			testKeys: []types.InstanceKey{keys.key1, keys.key2},
			expected: []types.FeatureInstance{
				{
					InstanceKey: keys.key1,
					Value:       &types.IntFeature{Val: 1},
				},
				{
					InstanceKey: keys.key2,
					Value:       &types.IntFeature{Val: 2},
				},
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			_, _, err := tt.builder.Build(context.Background())
			if (err != nil) != tt.buildErr {
				t.Errorf("Build() error = %v, wantErr %v", err, tt.buildErr)
			}
			getter, _, err := tt.builder.Build(context.Background())
			if (err != nil) != tt.buildErr {
				t.Errorf("Build() error = %v, wantErr %v", err, tt.buildErr)
			}

			if err == nil {
				for _, fi := range tt.expected {
					val, err := getter.Get(fi.InstanceKey)
					if (err != nil) != tt.getErr {
						t.Errorf("FeatureGetter.Get() error = %v, wantErr %v", err, tt.getErr)
					}
					if !reflect.DeepEqual(val.Value(), fi.Value.Value()) {
						t.Errorf("Get() got = %v, want %v", val.Value(), fi.Value.Value())
					}
				}
			}
		})
	}
}
