package source

import (
	"context"
	"fmt"
	"reflect"
	"strconv"
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/common/go_test_dynamo"

	"code.justin.tv/amzn/TwitchFeatureStoreClient/clients"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/fakes/metadatafake"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/metadata"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

const TestTable string = "test_table"

func newTestDDB() (dynamodbiface.DynamoDBAPI, error) {
	db := go_test_dynamo.Instance().Dynamo
	req := &dynamodb.CreateTableInput{
		AttributeDefinitions: []*dynamodb.AttributeDefinition{
			{
				AttributeName: aws.String("ofs_id"),
				AttributeType: aws.String("S"),
			},
		},
		KeySchema: []*dynamodb.KeySchemaElement{
			{
				AttributeName: aws.String("ofs_id"),
				KeyType:       aws.String("HASH"),
			},
		},
		ProvisionedThroughput: &dynamodb.ProvisionedThroughput{
			ReadCapacityUnits:  aws.Int64(5),
			WriteCapacityUnits: aws.Int64(5),
		},
		TableName: aws.String(TestTable),
	}
	_, err := db.CreateTableWithContext(context.Background(), req)
	return db, err
}

func loadTestDataBulk(db dynamodbiface.DynamoDBAPI, features ...*types.FeatureInstance) error {
	for i, f := range features {
		_, err := db.UpdateItem(buildDDBUpdateRequest(f.InstanceKey, func() *dynamodb.AttributeValue {
			return &dynamodb.AttributeValue{N: aws.String(fmt.Sprint(i))}
		}))
		if err != nil {
			return err
		}
	}

	return nil
}

func buildDDBUpdateRequest(k types.InstanceKey, a func() *dynamodb.AttributeValue) *dynamodb.UpdateItemInput {
	req := &dynamodb.UpdateItemInput{
		Key: map[string]*dynamodb.AttributeValue{
			"ofs_id": {S: aws.String(k.GetEntityStringKey())},
		},
		UpdateExpression: aws.String("SET #f = :val"),
		ExpressionAttributeNames: map[string]*string{
			"#f": aws.String(k.AttributeKey()),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":val": a(),
		},
		TableName: aws.String(TestTable),
	}
	return req
}

func generateFeatures(size int, key types.FeatureKey) []*types.FeatureInstance {
	f := make([]*types.FeatureInstance, 0)
	for i := 0; i < size; i++ {
		f = append(f, &types.FeatureInstance{
			InstanceKey: types.NewInstanceKey(key, []types.Entity{
				{
					Type: types.USER,
					Id:   fmt.Sprintf("test_user_%d", i),
				},
				{
					Type: types.CHANNEL,
					Id:   fmt.Sprintf("test_channel_%d", i),
				},
			}),
			Value: nil,
		})
	}
	return f
}

// TODO refactor this test to live outside of source package
func Test_dynamoDBOnlineSource_BulkGet_Parallelism(t *testing.T) {
	defer go_test_dynamo.Instance().Cleanup()
	// Setup test DDB table
	dbClient, err := newTestDDB()
	if err != nil {
		t.Errorf("unexpected error in test table creation:%v", err)
	}
	// Setup test feature data
	featureSet1 := generateFeatures(100, types.FeatureKey{
		FeatureID: "test_id_1",
		Namespace: "test",
		Version:   0,
	})
	featureSet2 := generateFeatures(10, types.FeatureKey{
		FeatureID: "test_id_2",
		Namespace: "test",
		Version:   0,
	})
	err = loadTestDataBulk(dbClient, append(featureSet1, featureSet2...)...)
	if err != nil {
		t.Errorf("unexpected error in load test table:%v", err)
	}

	type fields struct {
		DynamoDBSource metadata.DynamoDBSource
		namespace      string
		client         clients.DDBSubset
		metadata       map[types.FeatureKey]metadata.Provider
	}
	type args struct {
		ctx      context.Context
		features []*types.FeatureInstance
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{
			name: "success_scalar_integer",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					f1 := types.FeatureKey{
						FeatureID: "test_id_1",
						Namespace: "test",
						Version:   0,
					}
					f2 := types.FeatureKey{
						FeatureID: "test_id_2",
						Namespace: "test",
						Version:   0,
					}
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.INTEGER)
					p.GetValueDataShapeReturns(types.SCALAR)
					m[f1] = p
					m[f2] = p
					return m
				}(),
			},
			args: args{
				ctx:      context.Background(),
				features: append(featureSet1, featureSet2...),
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			d := &dynamoDBOnlineSource{
				DynamoDBSource: tt.fields.DynamoDBSource,
				namespace:      tt.fields.namespace,
				client:         tt.fields.client,
				metadata:       tt.fields.metadata,
				logger:         &types.EmptyLogger{},
			}
			if err := d.BulkGet(tt.args.ctx, tt.args.features); (err != nil) != tt.wantErr {
				t.Errorf("BulkGet() error = %v, wantErr %v", err, tt.wantErr)
			}

			for i, f := range featureSet1 {
				v := f.Value.Value().(int64)
				if v != int64(i) {
					t.Errorf("in feature set 1 got:%v, expected:%v", v, i)
				}
			}

			for i, f := range featureSet2 {
				v := f.Value.Value().(int64)
				if v != int64(i+100) {
					t.Errorf("in feature set 2 got:%v, expected:%v", v, i+100)
				}
			}
		})
	}
}

func Test_dynamoDBOnlineSource_BulkGet_FeatureTypes(t *testing.T) {
	defer go_test_dynamo.Instance().Cleanup()
	// Setup test DDB table
	dbClient, err := newTestDDB()
	if err != nil {
		t.Errorf("unexpected error in test table creation:%v", err)
	}
	// Setup test feature data
	f1 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 1),
			},
			{
				Type: types.CHANNEL,
				Id:   fmt.Sprintf("test_channel_%d", 1),
			},
		},
	}

	f2 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_2",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 1),
			},
			{
				Type: types.CHANNEL,
				Id:   fmt.Sprintf("test_channel_%d", 1),
			},
		},
	}

	type fields struct {
		DynamoDBSource metadata.DynamoDBSource
		namespace      string
		client         clients.DDBSubset
		metadata       map[types.FeatureKey]metadata.Provider
	}
	type args struct {
		ctx      context.Context
		features []*types.FeatureInstance
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		load    func() error
		expect  func(key types.InstanceKey) interface{}
		wantErr bool
	}{
		{
			name: "success_slice_integer",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.INTEGER)
					p.GetValueDataShapeReturns(types.LIST)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						L: []*dynamodb.AttributeValue{
							{N: aws.String(strconv.Itoa(1))},
							{N: aws.String(strconv.Itoa(2))},
							{N: aws.String(strconv.Itoa(3))},
						},
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						L: []*dynamodb.AttributeValue{
							{N: aws.String(strconv.Itoa(3))},
							{N: aws.String(strconv.Itoa(2))},
							{N: aws.String(strconv.Itoa(1))},
						},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}
				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return []int64{1, 2, 3}
				} else if key.FeatureKey == f2.FeatureKey {
					return []int64{3, 2, 1}
				}
				return nil
			},
			wantErr: false,
		},
		{
			name: "success_slice_float",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.FLOAT)
					p.GetValueDataShapeReturns(types.LIST)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						L: []*dynamodb.AttributeValue{
							{N: aws.String(fmt.Sprintf("%f", 1.1))},
							{N: aws.String(fmt.Sprintf("%f", 2.1))},
							{N: aws.String(fmt.Sprintf("%f", 3.1))},
						},
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						L: []*dynamodb.AttributeValue{
							{N: aws.String(fmt.Sprintf("%f", 3.1))},
							{N: aws.String(fmt.Sprintf("%f", 2.1))},
							{N: aws.String(fmt.Sprintf("%f", 1.1))},
						},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}
				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return []float64{1.1, 2.1, 3.1}
				} else if key.FeatureKey == f2.FeatureKey {
					return []float64{3.1, 2.1, 1.1}
				}
				return nil
			},
			wantErr: false,
		},
		{
			name: "success_slice_string",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.STRING)
					p.GetValueDataShapeReturns(types.VECTOR)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						L: []*dynamodb.AttributeValue{
							{S: aws.String("a")},
							{S: aws.String("b")},
							{S: aws.String("c")},
						},
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						L: []*dynamodb.AttributeValue{
							{S: aws.String("c")},
							{S: aws.String("b")},
							{S: aws.String("a")},
						},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}
				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return []string{"a", "b", "c"}
				} else if key.FeatureKey == f2.FeatureKey {
					return []string{"c", "b", "a"}
				}
				return nil
			},
			wantErr: false,
		},
		{
			name: "success_blob",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.STRING)
					p.GetValueDataShapeReturns(types.BLOB)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{1, 2, 3},
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{3, 2, 1},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}
				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return []byte{1, 2, 3}
				} else if key.FeatureKey == f2.FeatureKey {
					return []byte{3, 2, 1}
				}
				return nil
			},
			wantErr: false,
		},
		{
			name: "success_scalar_float",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.FLOAT)
					p.GetValueDataShapeReturns(types.SCALAR)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						N: aws.String(fmt.Sprintf("%f", 1.1)),
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						N: aws.String(fmt.Sprintf("%f", 2.1)),
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}
				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return float64(1.1)
				} else if key.FeatureKey == f2.FeatureKey {
					return float64(2.1)
				}
				return nil
			},
			wantErr: false,
		},
		{
			name: "success_scalar_string",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.STRING)
					p.GetValueDataShapeReturns(types.SCALAR)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						S: aws.String("a"),
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						S: aws.String("b"),
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}
				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return "a"
				} else if key.FeatureKey == f2.FeatureKey {
					return "b"
				}
				return nil
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			d := &dynamoDBOnlineSource{
				DynamoDBSource: tt.fields.DynamoDBSource,
				namespace:      tt.fields.namespace,
				client:         tt.fields.client,
				metadata:       tt.fields.metadata,
				logger:         &types.EmptyLogger{},
			}

			err := tt.load()
			if err != nil {
				t.Errorf("failed to load test data: %v", err)
			}

			if err := d.BulkGet(tt.args.ctx, tt.args.features); (err != nil) != tt.wantErr {
				t.Errorf("BulkGet() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			// f1
			if !reflect.DeepEqual(tt.expect(f1), tt.args.features[0].Value.Value()) {
				t.Errorf("Get() got = %+v, want %+v", tt.expect(f1), tt.args.features[0].Value)
				return
			}

			// f2
			if !reflect.DeepEqual(tt.expect(f2), tt.args.features[1].Value.Value()) {
				t.Errorf("Get() got = %+v, want %+v", tt.expect(f2), tt.args.features[1].Value)
				return
			}
		})
	}
}

// Test the case when both entity1 and entity2 have feature1 and feature2.
// But clients want to read feature 1 for entity1 and feature 2 for entity2.
// In the codes, we fetch a superset of attributes for items in the same
// BatchGet call, which results in feature 2 for entity1 also returns.
// Thus, we were having this issue.
// https://code.amazon.com/reviews/CR-45935572
//
//    ofs_id     | test_id_1@0 | test_id_1@1 |
// test_user_1     (requested)
// test_user_2                   (requested)
//
func Test_dynamoDBOnlineSource_BulkGet_FeatureVersions(t *testing.T) {
	defer go_test_dynamo.Instance().Cleanup()
	// Setup test DDB table
	dbClient, err := newTestDDB()
	if err != nil {
		t.Errorf("unexpected error in test table creation:%v", err)
	}
	// Setup test feature data
	f1 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 1),
			},
		},
	}

	f12 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   1,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 1),
			},
		},
	}

	f2 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   1,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 2),
			},
		},
	}

	f21 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 2),
			},
		},
	}

	type fields struct {
		DynamoDBSource metadata.DynamoDBSource
		namespace      string
		client         clients.DDBSubset
		metadata       map[types.FeatureKey]metadata.Provider
	}
	type args struct {
		ctx      context.Context
		features []*types.FeatureInstance
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		load    func() error
		expect  func(key types.InstanceKey) interface{}
		wantErr bool
	}{
		{
			name: "success_blob",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.STRING)
					p.GetValueDataShapeReturns(types.BLOB)
					m[f1.FeatureKey] = p
					m[f2.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f2,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{1, 2, 3},
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f12, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{1, 2, 3},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f2, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{3, 2, 1},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				req = buildDDBUpdateRequest(f21, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{3, 2, 1},
					}
				})
				_, err = dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return []byte{1, 2, 3}
				} else if key.FeatureKey == f2.FeatureKey {
					return []byte{3, 2, 1}
				}
				return nil
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			d := &dynamoDBOnlineSource{
				DynamoDBSource: tt.fields.DynamoDBSource,
				namespace:      tt.fields.namespace,
				client:         tt.fields.client,
				metadata:       tt.fields.metadata,
				logger:         &types.EmptyLogger{},
			}

			err := tt.load()
			if err != nil {
				t.Errorf("failed to load test data: %v", err)
				return
			}

			if err := d.BulkGet(tt.args.ctx, tt.args.features); (err != nil) != tt.wantErr {
				t.Errorf("BulkGet() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			// f1
			if !reflect.DeepEqual(tt.expect(f1), tt.args.features[0].Value.Value()) {
				t.Errorf("Get() got = %+v, want %+v", tt.expect(f1), tt.args.features[0].Value)
				return
			}

			// f2
			if !reflect.DeepEqual(tt.expect(f2), tt.args.features[1].Value.Value()) {
				t.Errorf("Get() got = %+v, want %+v", tt.expect(f2), tt.args.features[1].Value)
				return
			}
		})
	}
}

func Test_dynamoDBOnlineSource_BulkGet_MissingFeatures(t *testing.T) {
	defer go_test_dynamo.Instance().Cleanup()
	// Setup test DDB table
	dbClient, err := newTestDDB()
	if err != nil {
		t.Errorf("unexpected error in test table creation:%v", err)
	}
	// Setup test feature data
	f1 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 1),
			},
		},
	}

	f1_missing := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_2",
			Namespace: "test",
			Version:   1,
		},
		Entities: []types.Entity{
			{
				Type: types.USER,
				Id:   fmt.Sprintf("test_user_%d", 1),
			},
		},
	}

	type fields struct {
		DynamoDBSource metadata.DynamoDBSource
		namespace      string
		client         clients.DDBSubset
		metadata       map[types.FeatureKey]metadata.Provider
	}
	type args struct {
		ctx      context.Context
		features []*types.FeatureInstance
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		load    func() error
		expect  func(key types.InstanceKey) interface{}
		wantErr bool
	}{
		{
			name: "success_blob",
			fields: fields{
				DynamoDBSource: func() metadata.DynamoDBSource {
					s := &metadatafake.FakeDynamoDBSource{}
					s.GetTableReturns(TestTable)
					return s
				}(),
				namespace: "test",
				client:    dbClient,
				metadata: func() map[types.FeatureKey]metadata.Provider {
					m := make(map[types.FeatureKey]metadata.Provider)
					p := &metadatafake.FakeProvider{}
					p.GetValueDataTypeReturns(types.STRING)
					p.GetValueDataShapeReturns(types.BLOB)
					m[f1.FeatureKey] = p
					m[f1_missing.FeatureKey] = p
					return m
				}(),
			},
			args: args{
				ctx: context.Background(),
				features: []*types.FeatureInstance{
					{
						InstanceKey: f1,
						Value:       nil,
					},
					{
						InstanceKey: f1_missing,
						Value:       nil,
					},
				},
			},
			load: func() error {
				req := buildDDBUpdateRequest(f1, func() *dynamodb.AttributeValue {
					return &dynamodb.AttributeValue{
						B: []byte{1, 2, 3},
					}
				})
				_, err := dbClient.UpdateItem(req)
				if err != nil {
					return err
				}

				return nil
			},
			expect: func(key types.InstanceKey) interface{} {
				if key.FeatureKey == f1.FeatureKey {
					return []byte{1, 2, 3}
				}
				return nil
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			d := &dynamoDBOnlineSource{
				DynamoDBSource: tt.fields.DynamoDBSource,
				namespace:      tt.fields.namespace,
				client:         tt.fields.client,
				metadata:       tt.fields.metadata,
				logger:         &types.EmptyLogger{},
			}

			err := tt.load()
			if err != nil {
				t.Errorf("failed to load test data: %v", err)
				return
			}

			if err := d.BulkGet(tt.args.ctx, tt.args.features); (err != nil) != tt.wantErr {
				t.Errorf("BulkGet() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			// f1
			if !reflect.DeepEqual(tt.expect(f1), tt.args.features[0].Value.Value()) {
				t.Errorf("Get() got = %+v, want %+v", tt.expect(f1), tt.args.features[0].Value)
				return
			}

			// f2, feature value should be nil
			if !reflect.DeepEqual(tt.expect(f1_missing), tt.args.features[1].Value) {
				t.Errorf("Get() got = %+v, want %+v", tt.expect(f1_missing), tt.args.features[1].Value)
				return
			}
		})
	}
}
