package store

import (
	"context"
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/aws/request"
	"io/ioutil"

	"code.justin.tv/amzn/TwitchFeatureGo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/cep21/circuit/v3"
	"google.golang.org/protobuf/proto"
)

// S3Store is the client used for uploading and downloading objects from S3
type S3Store struct {
	S3       S3Api
	Bucket   string
	Circuits S3Circuits
}

//go:generate counterfeiter -o ../fakes/store/s3api.go . S3Api
type S3Api interface {
	GetObjectWithContext(aws.Context, *s3.GetObjectInput, ...request.Option) (*s3.GetObjectOutput, error)
}

// S3Circuits are the circuits for network operations on the s3 metadata store
type S3Circuits struct {
	GetCircuit *circuit.Circuit
}

var _ MetadataStore = &S3Store{}

func (s *S3Store) Get(ctx context.Context, key string) (*TwitchFeatureGo.FeatureMetadata, error) {
	var body []byte

	getErr := s.Circuits.GetCircuit.Run(ctx, func(ctx context.Context) error {
		object, err := s.S3.GetObjectWithContext(ctx, &s3.GetObjectInput{
			Bucket: aws.String(s.Bucket),
			Key:    aws.String(key),
		})
		if err != nil {
			return err
		}

		if object == nil {
			return errors.New("received a nil object from s3")
		}

		body, err = ioutil.ReadAll(object.Body)
		if err != nil {
			return fmt.Errorf("failed to read body of s3 object: %w", err)
		}
		return nil
	})
	if getErr != nil {
		return nil, getErr
	}

	var out TwitchFeatureGo.FeatureMetadata
	if err := proto.Unmarshal(body, &out); err != nil {
		return nil, fmt.Errorf("error on unmarshalling json: %w", err)
	}

	return &out, nil
}
