package store_test

import (
	"bytes"
	"context"
	"errors"
	"io/ioutil"
	"testing"

	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/cep21/circuit/v3"
	"google.golang.org/protobuf/proto"

	"code.justin.tv/amzn/TwitchFeatureGo"

	mocks "code.justin.tv/amzn/TwitchFeatureStoreClient/fakes/store"
	"code.justin.tv/amzn/TwitchFeatureStoreClient/store"
)

const dummyBucketName string = "dummy_bucket"
const dummyKey string = "dummy_key"

func createMockedS3ClientGet(t *testing.T, output *TwitchFeatureGo.FeatureMetadata, returnedError error) *mocks.FakeS3Api {
	t.Helper()

	var s3Output *s3.GetObjectOutput

	if output != nil {
		body, err := proto.Marshal(output)
		if err != nil {
			t.Fatal(err)
		}
		s3Output = &s3.GetObjectOutput{
			Body: ioutil.NopCloser(bytes.NewReader(body)),
		}
	}

	mockedS3 := new(mocks.FakeS3Api)
	mockedS3.GetObjectWithContextReturns(s3Output, returnedError)

	return mockedS3
}

func TestS3Store_Get(t *testing.T) {

	type fields struct {
		S3       *mocks.FakeS3Api
		Bucket   string
		Circuits store.S3Circuits
	}

	tests := []struct {
		name    string
		fields  fields
		want    *TwitchFeatureGo.FeatureMetadata
		wantErr bool
	}{
		{
			name: "retrieve data from s3",
			fields: fields{
				S3: createMockedS3ClientGet(t, &TwitchFeatureGo.FeatureMetadata{
					SchemaVersion: "1.0",
					Namespace:     "vxs",
				}, nil),
				Bucket: dummyBucketName,
				Circuits: store.S3Circuits{
					GetCircuit: &circuit.Circuit{},
				},
			},
			want: &TwitchFeatureGo.FeatureMetadata{
				SchemaVersion: "1.0",
				Namespace:     "vxs",
			},
			wantErr: false,
		},
		{
			name: "error from s3",
			fields: fields{
				S3:     createMockedS3ClientGet(t, nil, errors.New("s3 error")),
				Bucket: dummyBucketName,
				Circuits: store.S3Circuits{
					GetCircuit: &circuit.Circuit{},
				},
			},
			want:    nil,
			wantErr: true,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			s := &store.S3Store{
				S3:       tt.fields.S3,
				Bucket:   tt.fields.Bucket,
				Circuits: tt.fields.Circuits,
			}
			returnedOutput, err := s.Get(context.Background(), dummyKey)

			if (err != nil) != tt.wantErr {
				t.Errorf("Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			if !(tt.want == nil && tt.wantErr) && !proto.Equal(returnedOutput, tt.want) {
				t.Errorf("Get() got = %+v, want %+v", returnedOutput, tt.want)
			}
		})
	}
}
