package types

import (
	"fmt"
)

// ErrorFeatureNotExist indicates either the requested feature value is missing the feature source
// or FSC failed to retrieve the feature from the feature source
type ErrorFeatureNotExist struct {
	err string
}

func (e *ErrorFeatureNotExist) Error() string {
	return e.err
}

// ErrorUnexpected indicates an unexpected error happened in FSC. It can be caused by misconfigurations
// or bugs in FSC.
type ErrorUnexpected struct {
	err error
}

func (e *ErrorUnexpected) Error() string {
	return e.err.Error()
}

type FeatureGetter interface {
	Get(key InstanceKey) (TypedFeatureValue, error)
}

type InstanceAccessor interface {
	FeatureGetter
	GetInstance(key InstanceKey) (*FeatureInstance, error)
	SetInstance(key InstanceKey, value FeatureValue)
}

// maps feature key -> (a feature is defined on unique entities) entity string key -> feature instance
type instanceIndexer struct {
	allInstances map[FeatureKey]map[string]*FeatureInstance
}

var _ InstanceAccessor = &instanceIndexer{}
var _ FeatureGetter = &instanceIndexer{}

func NewInstanceIndexer() *instanceIndexer {
	return &instanceIndexer{
		allInstances: make(map[FeatureKey]map[string]*FeatureInstance),
	}
}

func (i *instanceIndexer) Get(key InstanceKey) (TypedFeatureValue, error) {
	instance, err := i.GetInstance(key)
	if err != nil {
		return nil, &ErrorUnexpected{err: err}
	}
	if instance.Value == nil {
		return nil, &ErrorFeatureNotExist{err: "feature is missing"}
	}
	return &TypedFeatureValueImpl{instance.Value}, nil
}

func (i *instanceIndexer) GetInstance(key InstanceKey) (*FeatureInstance, error) {
	if instanceMap, ok := i.allInstances[key.FeatureKey]; ok {
		if feature, ok := instanceMap[key.GetEntityStringKey()]; ok {
			return feature, nil
		}
	}
	return nil, fmt.Errorf("could not find feature instance for given instance key: %+v", key)
}

func (i *instanceIndexer) SetInstance(key InstanceKey, value FeatureValue) {
	instanceMap, ok := i.allInstances[key.FeatureKey]
	if !ok {
		instanceMap = make(map[string]*FeatureInstance)
	}
	instanceMap[key.GetEntityStringKey()] = &FeatureInstance{
		InstanceKey: key,
		Value:       value,
	}
	i.allInstances[key.FeatureKey] = instanceMap
}
