package types_test

import (
	"reflect"
	"testing"

	"code.justin.tv/amzn/TwitchFeatureStoreClient/types"
)

func Test_FeatureGetter(t *testing.T) {
	type Pair struct {
		key   types.InstanceKey
		value types.FeatureValue
	}

	testKey1 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.CATEGORY,
				Id:   "test_category_1",
			},
			{
				Type: types.USER,
				Id:   "test_user_1",
			},
		},
	}
	testKey2 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_2",
			Namespace: "test",
			Version:   1,
		},
		Entities: []types.Entity{
			{
				Type: types.QUERY,
				Id:   "test_query_2",
			},
		},
	}

	tests := []struct {
		name    string
		input   []Pair
		query   types.InstanceKey
		want    types.FeatureValue
		wantErr bool
	}{
		{
			name:    "fail - empty indexer",
			input:   nil,
			query:   testKey1,
			wantErr: true,
		},
		{
			name: "fail - missing keys",
			input: []Pair{
				{
					key:   testKey1,
					value: &types.IntFeature{Val: 100},
				},
			},
			query:   testKey2,
			wantErr: true,
		},
		{
			name: "success - nil FeatureValue",
			input: []Pair{
				{
					key:   testKey1,
					value: nil,
				},
			},
			query:   testKey1,
			want:    nil,
			wantErr: true,
		},
		{
			name: "success - multiple keys",
			input: []Pair{
				{
					key:   testKey1,
					value: &types.IntFeature{Val: 100},
				},
				{
					key:   testKey2,
					value: &types.StringFeature{Val: "test_value"},
				},
			},
			query:   testKey1,
			want:    &types.IntFeature{Val: 100},
			wantErr: false,
		},
		{
			name: "success - override keys",
			input: []Pair{
				{
					key:   testKey1,
					value: &types.IntFeature{Val: 100},
				},
				{
					key:   testKey1,
					value: &types.StringFeature{Val: "test_value"},
				},
			},
			query:   testKey1,
			want:    &types.StringFeature{Val: "test_value"},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ii := types.NewInstanceIndexer()
			for _, p := range tt.input {
				ii.SetInstance(p.key, p.value)
			}

			got, err := ii.Get(tt.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !tt.wantErr && !reflect.DeepEqual(got.Value(), tt.want.Value()) {
				t.Errorf("Get() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_InstanceIndexer(t *testing.T) {
	type Pair struct {
		key   types.InstanceKey
		value types.FeatureValue
	}

	testKey1 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_1",
			Namespace: "test",
			Version:   0,
		},
		Entities: []types.Entity{
			{
				Type: types.CATEGORY,
				Id:   "test_category_1",
			},
			{
				Type: types.USER,
				Id:   "test_user_1",
			},
		},
	}
	testKey2 := types.InstanceKey{
		FeatureKey: types.FeatureKey{
			FeatureID: "test_id_2",
			Namespace: "test",
			Version:   1,
		},
		Entities: []types.Entity{
			{
				Type: types.QUERY,
				Id:   "test_query_2",
			},
		},
	}

	tests := []struct {
		name    string
		input   []Pair
		query   types.InstanceKey
		want    *types.FeatureInstance
		wantErr bool
	}{
		{
			name:    "fail - empty indexer",
			input:   nil,
			query:   testKey1,
			wantErr: true,
		},
		{
			name: "fail - missing keys",
			input: []Pair{
				{
					key:   testKey1,
					value: &types.IntFeature{Val: 100},
				},
			},
			query:   testKey2,
			wantErr: true,
		},
		{
			name: "success - nil FeatureValue",
			input: []Pair{
				{
					key:   testKey1,
					value: nil,
				},
			},
			query: testKey1,
			want: &types.FeatureInstance{
				InstanceKey: testKey1,
				Value:       nil,
			},
			wantErr: false,
		},
		{
			name: "success - multiple keys",
			input: []Pair{
				{
					key:   testKey1,
					value: &types.IntFeature{Val: 100},
				},
				{
					key:   testKey2,
					value: &types.StringFeature{Val: "test_value"},
				},
			},
			query: testKey1,
			want: &types.FeatureInstance{
				InstanceKey: testKey1,
				Value:       &types.IntFeature{Val: 100},
			},
			wantErr: false,
		},
		{
			name: "success - override keys",
			input: []Pair{
				{
					key:   testKey1,
					value: &types.IntFeature{Val: 100},
				},
				{
					key:   testKey1,
					value: &types.StringFeature{Val: "test_value"},
				},
			},
			query: testKey1,
			want: &types.FeatureInstance{
				InstanceKey: testKey1,
				Value:       &types.StringFeature{Val: "test_value"},
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ii := types.NewInstanceIndexer()
			for _, p := range tt.input {
				ii.SetInstance(p.key, p.value)
			}

			got, err := ii.GetInstance(tt.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Get() got = %v, want %v", got, tt.want)
			}
		})
	}
}
