package types

import (
	"fmt"
	"strconv"
	"strings"
)

// A Feature identifies a unique feature defined in the Feature Registry
type FeatureKey struct {
	// The feature_id should match what is defined in the feature metadata
	FeatureID string
	Namespace string
	Version   int32
}

func NewFeatureKey(attributeKey, namespace string) (FeatureKey, error) {
	fk := FeatureKey{
		Namespace: namespace,
	}
	keys := strings.Split(attributeKey, "@")
	if len(keys) != 2 {
		return fk, fmt.Errorf("invalid feature attribute key:%s", attributeKey)
	}
	fk.FeatureID = keys[0]
	v, err := strconv.Atoi(keys[1])
	if err != nil {
		return fk, fmt.Errorf("invalid feature version number:%s", keys[1])
	}
	fk.Version = int32(v)

	return fk, nil
}

func (i *FeatureKey) GetPath() string {
	return fmt.Sprintf("%v/%v/%v/metadata", i.Namespace, i.FeatureID, i.Version)
}

func (i *FeatureKey) AttributeKey() string {
	return fmt.Sprintf("%s@%d", i.FeatureID, i.Version)
}
