package types

import (
	"fmt"
	"strings"
)

type EntityType string

const (
	OTHER    EntityType = "OTHER"
	USER     EntityType = "USER"
	DEVICE   EntityType = "DEVICE"
	CHANNEL  EntityType = "CHANNEL"
	QUERY    EntityType = "QUERY"
	CATEGORY EntityType = "CATEGORY"
)

// An Entity should uniquely identify an instance.
type Entity struct {
	Type EntityType
	Id   string
}

// A InstanceKey can uniquely identify an instance of a feature.
// It's preferred to use NewInstanceKey function to create the instance key.
type InstanceKey struct {
	FeatureKey
	Entities        []Entity
	entityStringKey string
	cacheKey        string
}

func NewInstanceKey(featureKey FeatureKey, entities []Entity) InstanceKey {
	entityStringKey := generateEntityStringKey(entities)
	return InstanceKey{
		FeatureKey:      featureKey,
		Entities:        entities,
		entityStringKey: entityStringKey,
		cacheKey:        generateCacheKey(featureKey, entityStringKey),
	}
}

func generateEntityStringKey(entities []Entity) string {
	if len(entities) == 0 {
		return ""
	}
	var b strings.Builder
	for _, e := range entities {
		fmt.Fprintf(&b, "%s#", e.Id)
	}
	return b.String()[:b.Len()-1]
}

// Return a string which is formed by concatenating the entity ids in order.
// E.g. for Entities [{Type: USER, Id: 123}, {Type: CHANNEL, Id: 2314}], the
// string key should be "123#2314"
func (i *InstanceKey) GetEntityStringKey() string {
	if len(i.entityStringKey) > 0 {
		return i.entityStringKey
	}
	return generateEntityStringKey(i.Entities)
}

func generateCacheKey(featureKey FeatureKey, entityStringKey string) string {
	return strings.Join([]string{featureKey.FeatureID, featureKey.Namespace, fmt.Sprintf("%v", featureKey.Version), entityStringKey}, ":")
}

func (i *InstanceKey) GetCacheKey() string {
	if len(i.cacheKey) > 0 {
		return i.cacheKey
	}
	return generateCacheKey(i.FeatureKey, i.GetEntityStringKey())
}

// FeatureInstance is a feature key-value pair.
type FeatureInstance struct {
	InstanceKey
	Value FeatureValue
}
