package types

import (
	"fmt"
)

type ValueDataType int

const (
	STRING ValueDataType = iota
	INTEGER
	FLOAT
)

type ValueDataShape int

const (
	SCALAR ValueDataShape = iota
	VECTOR
	LIST
	BLOB
)

type TypedFeatureValue interface {
	FeatureValue
	Int() (int64, error)
	Float() (float64, error)
	String() (string, error)
	IntSlice() ([]int64, error)
	FloatSlice() ([]float64, error)
	StringSlice() ([]string, error)
	StringToIntMap() (map[string]int64, error)
	StringToFloatMap() (map[string]float64, error)
	StringToStringMap() (map[string]string, error)
	Blob() ([]byte, error)
}

type TypedFeatureValueImpl struct {
	FValue FeatureValue
}

var _ TypedFeatureValue = &TypedFeatureValueImpl{}

func (c *TypedFeatureValueImpl) Value() interface{} {
	return c.FValue.Value()
}

func (c *TypedFeatureValueImpl) IsDefault() bool {
	return c.FValue.IsDefault()
}

func (c *TypedFeatureValueImpl) Int() (int64, error) {
	if v, ok := c.Value().(int64); ok {
		return v, nil
	}
	return 0, fmt.Errorf("incompatible feature value type: expected int64 got %+v", c.Value())
}

func (c *TypedFeatureValueImpl) Float() (float64, error) {
	if v, ok := c.Value().(float64); ok {
		return v, nil
	}
	return 0, fmt.Errorf("incompatible feature value type: expected float64 got %+v", c.Value())
}

func (c *TypedFeatureValueImpl) String() (string, error) {
	if v, ok := c.Value().(string); ok {
		return v, nil
	}
	return "", fmt.Errorf("incompatible feature value type: expected string got %+v", c.Value())
}

func (c *TypedFeatureValueImpl) IntSlice() ([]int64, error) {
	if v, ok := c.Value().([]int64); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected []int64 got %+v", c.Value())
}

func (c *TypedFeatureValueImpl) FloatSlice() ([]float64, error) {
	if v, ok := c.Value().([]float64); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected []float64 got %+v", c.Value())
}

func (c *TypedFeatureValueImpl) StringSlice() ([]string, error) {
	if v, ok := c.Value().([]string); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected []string got %+v", c.Value())
}

func (c *TypedFeatureValueImpl) StringToIntMap() (map[string]int64, error) {
	if v, ok := c.Value().(map[string]int64); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected map[string]int64 got %T", c.Value())
}

func (c *TypedFeatureValueImpl) StringToFloatMap() (map[string]float64, error) {
	if v, ok := c.Value().(map[string]float64); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected map[string]float64 got %T", c.Value())
}

func (c *TypedFeatureValueImpl) StringToStringMap() (map[string]string, error) {
	if v, ok := c.Value().(map[string]string); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected map[string]string got %T", c.Value())
}

func (c *TypedFeatureValueImpl) Blob() ([]byte, error) {
	if v, ok := c.Value().([]byte); ok {
		return v, nil
	}
	return nil, fmt.Errorf("incompatible feature value type: expected []byte got %+v", c.Value())
}

// FeatureValue represents different types of the feature value.
// It also includes information about if the value is a default or not.
// e.g. number, string
type FeatureValue interface {
	Value() interface{}
	IsDefault() bool
}

type IntFeature struct {
	Val     int64
	Default bool
}

func (f *IntFeature) Value() interface{} {
	return f.Val
}

func (f *IntFeature) IsDefault() bool {
	return f.Default
}

type FloatFeature struct {
	Val     float64
	Default bool
}

func (f *FloatFeature) Value() interface{} {
	return f.Val
}

func (f *FloatFeature) IsDefault() bool {
	return f.Default
}

type StringFeature struct {
	Val     string
	Default bool
}

func (f *StringFeature) Value() interface{} {
	return f.Val
}

func (f *StringFeature) IsDefault() bool {
	return f.Default
}

type StringSliceFeature struct {
	Val     []string
	Default bool
}

func (f *StringSliceFeature) Value() interface{} {
	return f.Val
}

func (f *StringSliceFeature) IsDefault() bool {
	return f.Default
}

type IntSliceFeature struct {
	Val     []int64
	Default bool
}

func (f *IntSliceFeature) Value() interface{} {
	return f.Val
}

func (f *IntSliceFeature) IsDefault() bool {
	return f.Default
}

type FloatSliceFeature struct {
	Val     []float64
	Default bool
}

func (f *FloatSliceFeature) Value() interface{} {
	return f.Val
}

func (f *FloatSliceFeature) IsDefault() bool {
	return f.Default
}

type StringToIntMapFeature struct {
	Val     map[string]int64
	Default bool
}

func (f *StringToIntMapFeature) Value() interface{} {
	return f.Val
}

func (f *StringToIntMapFeature) IsDefault() bool {
	return f.Default
}

type StringToFloatMapFeature struct {
	Val     map[string]float64
	Default bool
}

func (f *StringToFloatMapFeature) Value() interface{} {
	return f.Val
}

func (f *StringToFloatMapFeature) IsDefault() bool {
	return f.Default
}

type StringToStringMapFeature struct {
	Val     map[string]string
	Default bool
}

func (f *StringToStringMapFeature) Value() interface{} {
	return f.Val
}

func (f *StringToStringMapFeature) IsDefault() bool {
	return f.Default
}

type BlobFeature struct {
	Val     []byte
	Default bool
}

func (f *BlobFeature) Value() interface{} {
	return f.Val
}

func (f *BlobFeature) IsDefault() bool {
	return f.Default
}
