package types

import (
	"reflect"
	"testing"
)

func Test_TypedFeatureValueImpl_Blob(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    []byte
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&BlobFeature{Val: []byte{1, 2, 3}}},
			want:    []byte{1, 2, 3},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&StringFeature{Val: "test"}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.Blob()
			if (err != nil) != tt.wantErr {
				t.Errorf("Blob() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Blob() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_Float(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    float64
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&FloatFeature{Val: 1.1}},
			want:    1.1,
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&FloatSliceFeature{Val: []float64{1.1, 2.2, 3.3}}},
			want:    0,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.Float()
			if (err != nil) != tt.wantErr {
				t.Errorf("Float() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("Float() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_FloatSlice(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    []float64
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&FloatSliceFeature{Val: []float64{1.1, 2.2, 3.3}}},
			want:    []float64{1.1, 2.2, 3.3},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&FloatFeature{Val: 1.1}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.FloatSlice()
			if (err != nil) != tt.wantErr {
				t.Errorf("FloatSlice() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("FloatSlice() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_Int(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    int64
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&IntFeature{Val: 1}},
			want:    1,
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&IntSliceFeature{Val: []int64{1, 2, 3}}},
			want:    0,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.Int()
			if (err != nil) != tt.wantErr {
				t.Errorf("Int() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("Int() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_IntSlice(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    []int64
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&IntSliceFeature{Val: []int64{1, 2, 3}}},
			want:    []int64{1, 2, 3},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&IntFeature{Val: 1}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.IntSlice()
			if (err != nil) != tt.wantErr {
				t.Errorf("IntSlice() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("IntSlice() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_String(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    string
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&StringFeature{Val: "test"}},
			want:    "test",
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&StringSliceFeature{Val: []string{"test"}}},
			want:    "",
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.String()
			if (err != nil) != tt.wantErr {
				t.Errorf("String() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("String() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_StringSlice(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    []string
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&StringSliceFeature{Val: []string{"a", "b", "c"}}},
			want:    []string{"a", "b", "c"},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&StringFeature{Val: "a"}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.StringSlice()
			if (err != nil) != tt.wantErr {
				t.Errorf("StringSlice() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("StringSlice() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_StringToIntMap(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    map[string]int64
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&StringToIntMapFeature{Val: map[string]int64{"a": 1, "b": 2, "c": 3}}},
			want:    map[string]int64{"a": 1, "b": 2, "c": 3},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&StringFeature{Val: "a"}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.StringToIntMap()
			if (err != nil) != tt.wantErr {
				t.Errorf("StringToIntMap() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("StringToIntMap() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_StringToFloatMap(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    map[string]float64
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&StringToFloatMapFeature{Val: map[string]float64{"a": 1.1, "b": 2.2, "c": 3.3}}},
			want:    map[string]float64{"a": 1.1, "b": 2.2, "c": 3.3},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&StringFeature{Val: "a"}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.StringToFloatMap()
			if (err != nil) != tt.wantErr {
				t.Errorf("StringToFloatMap() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("StringToFloatMap() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_StringToStringMap(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name    string
		fields  fields
		want    map[string]string
		wantErr bool
	}{
		{
			name:    "success",
			fields:  fields{&StringToStringMapFeature{Val: map[string]string{"a": "true", "b": "false", "c": "true"}}},
			want:    map[string]string{"a": "true", "b": "false", "c": "true"},
			wantErr: false,
		},
		{
			name:    "fail",
			fields:  fields{&StringFeature{Val: "a"}},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			got, err := c.StringToStringMap()
			if (err != nil) != tt.wantErr {
				t.Errorf("StringToStringMap() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("StringToStringMap() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_TypedFeatureValueImpl_Value(t *testing.T) {
	type fields struct {
		FValue FeatureValue
	}
	tests := []struct {
		name   string
		fields fields
		want   interface{}
	}{
		{
			name:   "success",
			fields: fields{&StringSliceFeature{Val: []string{"a", "b", "c"}}},
			want:   []string{"a", "b", "c"},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &TypedFeatureValueImpl{
				FValue: tt.fields.FValue,
			}
			if got := c.Value(); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Value() = %v, want %v", got, tt.want)
			}
		})
	}
}
