package requestlog

import (
	logging "code.justin.tv/amzn/TwitchLogging"
	"code.justin.tv/hygienic/messagebatch"
	"code.justin.tv/hygienic/messagebatch/ext/cwlogevent"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
	"github.com/cep21/circuit/v3"
	"github.com/pkg/errors"
)

type logWrapper struct {
	prefix string
	logger logging.Logger
}

func (d *logWrapper) Log(keyvals ...interface{}) {
	d.logger.Log(d.prefix, keyvals...)
}

// initializeBatcher initialises a CloudwatchLogBatcher.
func initializeBatcher(twitchLogger logging.Logger, cwCircuit *circuit.Circuit, logGroupName, region string) (cwlogevent.CloudwatchLogBatcher, error) {
	awsConf := &aws.Config{
		// Current region is pre-defined in the execution environment
		Region:              aws.String(region),
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	}

	sess, err := session.NewSession(awsConf)
	if err != nil {
		return cwlogevent.CloudwatchLogBatcher{}, errors.Wrap(err, "failed to start AWS session")
	}

	logWrapper := &logWrapper{
		prefix: "requestLogger",
		logger: twitchLogger,
	}

	batcher := cwlogevent.CloudwatchLogBatcher{
		Batcher: messagebatch.Batcher{
			Log:    logWrapper,
			Events: make(chan interface{}, 1000),
		},
		Config: &cwlogevent.Config{
			LogGroupName: logGroupName,
		},
		Client:  cloudwatchlogs.New(sess, awsConf),
		Circuit: cwCircuit,
	}

	err = batcher.Setup()
	if err != nil {
		return batcher, err
	}

	go func() {
		err := batcher.Start()
		if err != nil {
			logWrapper.Log(err)
		}
	}()

	return batcher, nil
}
