package requestlog

import (
	"context"
	"encoding/json"
	"sync"

	"google.golang.org/protobuf/proto"
)

// ContextTrace is a container for additional request information.
type ContextTrace struct {
	Body         proto.Message
	CallerSvc    string
	CustomFields interface{}

	mu       sync.Mutex
	ints     map[string]int64
	strings  map[string]string
	floats   map[string]float64
	forceLog bool
}

// SetTraceInt adds an int to the context trace, if it exists
func SetTraceInt(ctx context.Context, key string, value int64) {
	trace := GetTrace(ctx)

	trace.mu.Lock()
	defer trace.mu.Unlock()

	if trace.ints == nil {
		trace.ints = map[string]int64{}
	}

	trace.ints[key] = value
}

// SetTraceString adds a string to the context trace, if it exists
func SetTraceString(ctx context.Context, key string, value string) {
	trace := GetTrace(ctx)

	trace.mu.Lock()
	defer trace.mu.Unlock()

	if trace.strings == nil {
		trace.strings = map[string]string{}
	}

	trace.strings[key] = value
}

// SetTraceFloat adds a float to the context trace, if it exists
func SetTraceFloat(ctx context.Context, key string, value float64) {
	trace := GetTrace(ctx)

	trace.mu.Lock()
	defer trace.mu.Unlock()

	if trace.floats == nil {
		trace.floats = map[string]float64{}
	}

	trace.floats[key] = value
}

// MarshalJSON implements custom marshalling for ContextTrace to expose the hidden fields
func (t *ContextTrace) MarshalJSON() ([]byte, error) {
	t.mu.Lock()
	defer t.mu.Unlock()

	return json.Marshal(&struct {
		Body         proto.Message      `json:"body,omitempty"`
		CallerSvc    string             `json:"caller_svc,omitempty"`
		CustomFields interface{}        `json:"custom_fields,omitempty"`
		Ints         map[string]int64   `json:"ints,omitempty"`
		Strings      map[string]string  `json:"strings,omitempty"`
		Floats       map[string]float64 `json:"floats,omitempty"`
	}{
		Body:         t.Body,
		CallerSvc:    t.CallerSvc,
		CustomFields: t.CustomFields,
		Ints:         t.ints,
		Strings:      t.strings,
		Floats:       t.floats,
	})
}

// GetTrace returns a ContextTrace object which will potentially contain trace data.
func GetTrace(ctx context.Context) *ContextTrace {
	ret := getTraceOrNil(ctx)
	if ret == nil {
		// Return a dummy object so users don't have to worry about nil panics
		return &ContextTrace{}
	}

	return ret
}

// EnsureTrace returns context with an empty Trace if it does not exist, or the original Trace if it exists.
func EnsureTrace(ctx context.Context) context.Context {
	ctxTrace := getTraceOrNil(ctx)
	if ctxTrace == nil {
		return withTrace(ctx)
	}

	return ctx
}

// ForceLog ensures that the request related to ctx will emit a log event.
// Is a no-op if a context trace is not found on the context.
func ForceLog(ctx context.Context) {
	if ctxTrace := getTraceOrNil(ctx); ctxTrace != nil {
		ctxTrace.mu.Lock()
		ctxTrace.forceLog = true
		ctxTrace.mu.Unlock()
	}
}

// getTraceOrNil returns a ContextTrace object if found in the context, otherwise nil.
func getTraceOrNil(ctx context.Context) *ContextTrace {
	ret, exists := ctx.Value(ctxKeyBody).(*ContextTrace)
	if exists {
		return ret
	}

	return nil
}
