package requestlog

import (
	"context"
	"strconv"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"github.com/twitchtv/twirp"
)

// EventHook provides a twirp hook for sending request log events to Cloudwatch
func EventHook(requestLogger *Logger) *twirp.ServerHooks {
	return &twirp.ServerHooks{
		Error: func(ctx context.Context, err twirp.Error) context.Context {
			ctx = WithEventError(ctx, err)
			return ctx
		},
		RequestReceived: func(ctx context.Context) (context.Context, error) {
			ctx = WithEventStart(ctx)
			return ctx, nil
		},
		ResponsePrepared: WithEventHandled,
		ResponseSent: func(ctx context.Context) {
			method, _ := twirp.MethodName(ctx)
			code, _ := twirp.StatusCode(ctx)
			requestLogger.logEventImpl(ctx, method, code)
		},
	}
}

// SLOHook provides a twirp hook for reporting SLOs with Telemetry.
func SLOHook(reporter telemetry.SampleReporter, requestLogger *Logger) *twirp.ServerHooks {
	return &twirp.ServerHooks{
		ResponseSent: func(ctx context.Context) {
			sloPassed := true

			method, ok := twirp.MethodName(ctx)
			if !ok {
				return // we shouldn't report SLO if there's no method
			}

			if code, ok := twirp.StatusCode(ctx); ok {
				if codeInt, err := strconv.Atoi(code); err == nil && codeInt >= 500 {
					sloPassed = false
				}
			}

			if recvTime, ok := ctx.Value(eventStartTimeKey).(time.Time); ok {
				duration := time.Since(recvTime)
				durationMS := duration.Nanoseconds() / time.Millisecond.Nanoseconds()
				sloPassed = sloPassed && requestLogger.passesSLO(durationMS, method)
			}

			reportSLO(reporter, method, sloPassed)
		},
	}
}

func reportSLO(reporter telemetry.SampleReporter, methodName string, passed bool) {
	// need to copy the reporter to add "Operation" to get the standard rollups
	allDims := make(map[string]string, len(reporter.Dimensions)+1)

	for k, v := range reporter.Dimensions {
		allDims[k] = v
	}

	if methodName != "" {
		allDims["Operation"] = methodName
	}

	reporter.Dimensions = allDims
	percentValue := 0.0

	if passed {
		reporter.Report("SLOPassed", 1, telemetry.UnitCount)

		percentValue = 1.0
	} else {
		reporter.Report("SLOFailed", 1, telemetry.UnitCount)
	}

	reporter.Report("SLOPercent", percentValue, telemetry.UnitCount)
}
