package requestlog

import (
	"math/rand"
	"testing"
	"time"
)

//nolint:funlen // test cases
func TestShouldSend(t *testing.T) {
	tests := []struct {
		name   string
		input  *event
		logger *Logger
		want   bool
	}{
		{
			"always allows write events",
			&event{Method: "write-endpoint", DurationMS: 10},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 1,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			true,
		},
		{
			"allows if method missing",
			&event{DurationMS: 10},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 0,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			true,
		},
		{
			"always allows long duration events",
			&event{Method: "read-endpoint", DurationMS: 10000},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 1,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			true,
		},
		{
			"sometimes allows short read events",
			&event{Method: "read-endpoint", DurationMS: 10},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 1,
				sampleRates:       map[string]float64{"read-endpoint": 1.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			true,
		},
		{
			"sometimes disallows short read events",
			&event{Method: "read-endpoint", DurationMS: 10},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 1,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			false,
		},
		{
			"always allows logging if log forced on, even for short read events",
			&event{
				ContextTrace: &ContextTrace{forceLog: true},
				Method:       "read-endpoint",
				DurationMS:   10,
			},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 1,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			true,
		},
		{
			"always allows errors",
			&event{Method: "read-endpoint", DurationMS: 10, StatusCode: "500", Error: "help"},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 1,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
			},
			true,
		},
		{
			"allows client errors, if SampleClientErrors is not set",
			&event{Method: "read-endpoint", DurationMS: 10, StatusCode: "400", Error: "help"},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 0,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
				sampleClientErrors: false,
			},
			true,
		},
		{
			"disallows client errors, if SampleClientErrors is set",
			&event{Method: "read-endpoint", DurationMS: 10, StatusCode: "400", Error: "help"},
			&Logger{
				rand:              rand.New(rand.NewSource(1)),
				defaultSampleRate: 0,
				sampleRates:       map[string]float64{"read-endpoint": 0.0},
				methodSLOs: map[string]time.Duration{
					"read-endpoint":  1000 * time.Millisecond,
					"write-endpoint": 1000 * time.Millisecond,
				},
				sampleClientErrors: true,
			},
			false,
		},
	}

	for _, tt := range tests {
		got := tt.logger.shouldSend(tt.input)

		if got != tt.want {
			t.Errorf("wrong output [%v]:\ngot:  %v\nwant: %v", tt.name, got, tt.want)
		}
	}
}

func Test_Logger_passesSLO_NoSLOs(t *testing.T) {
	l := &Logger{}

	if !l.passesSLO(100, "UnknownMethod") {
		t.Errorf("unknown method should be assumed to pass SLO check")
	}
}
