package sessionsclient

import (
	"context"
	"net"
	"time"

	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "sessions"
)

// Metadata contains data about the creation of a session, such as the associated user agent
type Metadata struct {
	Location  *string `json:"location"`
	IP        net.IP  `json:"ip"`
	UserAgent *string `json:"user_agent"`
}

// Session represents an authorization of a client and associated metadata about the creation of the session
type Session struct {
	ClientID       string     `json:"client_id"`
	UserID         string     `json:"user_id"`
	CreatorID      string     `json:"creator_id"`
	ExpiresAt      *time.Time `json:"expires_at"`
	CreatedAt      *time.Time `json:"created_at"`
	UpdatedAt      *time.Time `json:"updated_at"`
	DeletedAt      *time.Time `json:"deleted_at"`
	TokenCleartext string     `json:"-"`
	Metadata       Metadata   `json:"metadata"`
	SpareKey       *string    `json:"spare_key_jwt"`
}

// Client is used to make API requests to sessions service
//go:generate mockery -name=Client
type Client interface {
	Validate(ctx context.Context, validateParams *ValidateSessionRequest) (*Session, error)
	Create(ctx context.Context, createParams *CreateSessionRequest) (string, error)
	Invalidate(ctx context.Context, invalidateParams *InvalidateSessionRequest) error
	InvalidateSessions(ctx context.Context, userID string, invalidateParams *InvalidateSessionsRequest) error
	DeleteSession(ctx context.Context, deleteParams *DeleteSessionRequest) error
	BulkDeleteSessions(ctx context.Context, deleteParams *BulkDeleteSessionsRequest) error
	ListSessions(ctx context.Context, userID string) ([]*Session, error)
}

type client struct {
	client *twitchclient.JSONClient
}

// NewClient initializes a client using the given config
func NewClient(conf twitchclient.ClientConf) Client {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	httpCli := twitchclient.NewHTTPClient(conf)
	twitchClient := twitchclient.NewJSONClient(conf.Host, httpCli)
	return &client{twitchClient}
}
