package sessionsclient

import (
	"context"
	"fmt"
	"strings"

	"code.justin.tv/foundation/twitchclient"
)

type CreateSessionResponse struct {
	Token string `json:"token"`
}

type CreateSessionRequest struct {
	UserID    string `json:"user_id"`
	CreatorID string `json:"creator_id"`
	UserAgent string `json:"user_agent"`
	IP        string `json:"ip"`
	Location  string `json:"location"`
	ClientID  string `json:"client_id"`
	ExpiresIn int    `json:"expires_in"`
	SpareKey  string `json:"spare_key_jwt"`
}

func (c *client) Create(ctx context.Context, params *CreateSessionRequest) (string, error) {
	var createResponse CreateSessionResponse
	if e := params.Validate(); e != nil {
		return "", e
	}
	err := c.client.Post(ctx, "/sessions", params, &createResponse)
	if err != nil {
		if e, ok := err.(*twitchclient.Error); ok {
			return "", parseErrorResponse(e)
		}
		return "", err
	}

	return createResponse.Token, nil
}

func (r *CreateSessionRequest) Validate() *ErrorResponse {
	missingParams := []string{}
	if r.UserID == "" {
		missingParams = append(missingParams, "user_id")
	}
	if r.CreatorID == "" {
		missingParams = append(missingParams, "creator_id")
	}
	if r.UserAgent == "" {
		missingParams = append(missingParams, "user_agent")
	}
	if r.IP == "" {
		missingParams = append(missingParams, "ip")
	}
	if r.Location == "" {
		missingParams = append(missingParams, "location")
	}
	if r.ClientID == "" {
		missingParams = append(missingParams, "client_id")
	}
	if r.ExpiresIn == 0 {
		missingParams = append(missingParams, "expires_in")
	}
	if r.SpareKey == "" {
		missingParams = append(missingParams, "spare_key_jwt")
	}

	if len(missingParams) > 0 {
		e := ErrMissingParams
		e.Message = fmt.Sprintf("Missing required parameters: %s", strings.Join(missingParams, ","))
		return e
	}

	return nil
}

func (r *CreateSessionRequest) IsImpersonationRequest() bool {
	return r.CreatorID != r.UserID
}
