package sessionsclient

import (
	"net/http"

	"code.justin.tv/foundation/twitchclient"
)

// All Errors used within sessions. ErrorMessage is populated
// in serveError
var (
	// Errors returned to client

	ErrInvalidToken = &ErrorResponse{
		Message:    "invalid session token",
		StatusCode: http.StatusNotFound,
	}
	ErrUserIDRequired = &ErrorResponse{
		Message:    "Must supply a non-empty user ID",
		StatusCode: http.StatusBadRequest,
	}
	ErrInvalidRequestParams = &ErrorResponse{
		Message:    "Invalid request parameters. Must have a non-nil request body",
		StatusCode: http.StatusBadRequest,
	}
	ErrRequestTooLarge = &ErrorResponse{
		Message:    "too many reqeusts in the batch, reduce the number of reqeust",
		StatusCode: http.StatusRequestEntityTooLarge,
	}
	ErrDatabaseUnexpected = &ErrorResponse{
		Message:    "unexpected error from database",
		StatusCode: http.StatusServiceUnavailable,
	}
	ErrSessionNotFound = &ErrorResponse{
		Message:    "session not found",
		StatusCode: http.StatusNotFound,
	}
	ErrNoRowsAffected = &ErrorResponse{
		Message:    "no rows affected",
		StatusCode: http.StatusUnprocessableEntity,
	}
	ErrUnhandledServerException = &ErrorResponse{
		Message:    "unhandled server exception",
		StatusCode: http.StatusInternalServerError,
	}
	ErrUnhandledClientException = &ErrorResponse{
		Message:    "unhandled client exception",
		StatusCode: http.StatusBadRequest,
	}
	ErrInvalidJson = &ErrorResponse{
		Message:    "invalid request. malformed json body",
		StatusCode: http.StatusBadRequest,
	}
	// Rendered in api/helpers.go
	ErrMissingParams = &ErrorResponse{
		Message:    "missing required parameters: xxx",
		StatusCode: http.StatusBadRequest,
	}

	// Errors used internally. These are not returned to the client

	ErrCacheMiss = &ErrorResponse{
		Message:    "cache miss",
		StatusCode: http.StatusNotFound,
	}
	ErrCacheUnexpected = &ErrorResponse{
		Message:    "unexpected error from cache",
		StatusCode: http.StatusFailedDependency,
	}
)

type ErrorResponse struct {
	// Retained for backwards compatibility with Web/Web.  Once Web/Web
	// is deprecated. ErrorMessage should be deprecated
	ErrorMessage string `json:"error"`
	// Same as ErrorMessage. Used for compatibility with TwitchClient
	Message    string `json:"message"`
	StatusCode int    `json:"-"`
}

func (e *ErrorResponse) Error() string {
	return e.Message
}

func parseErrorResponse(e *twitchclient.Error) *ErrorResponse {
	switch e.Message {
	case ErrInvalidToken.Error():
		return ErrInvalidToken
	case ErrUserIDRequired.Error():
		return ErrUserIDRequired
	case ErrInvalidRequestParams.Error():
		return ErrInvalidRequestParams
	case ErrRequestTooLarge.Error():
		return ErrRequestTooLarge
	case ErrDatabaseUnexpected.Error():
		return ErrDatabaseUnexpected
	case ErrSessionNotFound.Error():
		return ErrSessionNotFound
	case ErrNoRowsAffected.Error():
		return ErrNoRowsAffected
	case ErrUnhandledServerException.Error():
		return ErrUnhandledServerException
	case ErrUnhandledClientException.Error():
		return ErrUnhandledClientException
	case ErrInvalidJson.Error():
		return ErrInvalidJson
	default:
		err := ErrUnhandledClientException
		err.Message = e.Message
		err.StatusCode = e.StatusCode
		return err
	}
}

func IsBadRequest(err error) bool {
	if err == nil {
		return false
	}

	sessionsErr, ok := err.(*ErrorResponse)
	if !ok {
		return false
	}

	return sessionsErr.StatusCode >= 400 && sessionsErr.StatusCode < 500
}
