package sessionsclient

import (
	"context"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/foundation/twitchclient"
)

var badResponseCodes = []int{
	http.StatusInternalServerError,
	http.StatusBadRequest,
	http.StatusForbidden,
}

func TestInvalidateSessionsSuccess(t *testing.T) {
	ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusNoContent)
	}))
	defer ts.Close()
	o := NewClient(twitchclient.ClientConf{
		Host: ts.URL,
	})

	req := &InvalidateSessionsRequest{
		KeepSession: "1234::5678",
		ClientID:    "asdfjkl4ioj1234",
	}

	err := o.InvalidateSessions(context.Background(), "1234", req)
	assert.Nil(t, err, "Should have no error if the server returns a 204 No Content")
}

func TestInvalidateSessionsServerFailure(t *testing.T) {
	for _, code := range badResponseCodes {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(code)
		}))
		o := NewClient(twitchclient.ClientConf{
			Host: ts.URL,
		})

		req := &InvalidateSessionsRequest{
			KeepSession: "1234::5678",
			ClientID:    "asdfjkl4ioj1234",
		}

		err := o.InvalidateSessions(context.Background(), "1234", req)
		assert.Error(t, err, "Should have an error if the server returns a non-204 response")
		ts.Close()
	}
}

func TestInvalidateSessionsBadUserIDInput(t *testing.T) {
	o := NewClient(twitchclient.ClientConf{
		Host: "foo",
	})

	req := &InvalidateSessionsRequest{
		KeepSession: "1234::5678",
		ClientID:    "asdfjkl4ioj1234",
	}
	err := o.InvalidateSessions(context.Background(), "", req)
	assert.Equal(t, ErrUserIDRequired, err, "Should receive a User ID Required error")
}

func TestInvalidateSessionsNilInput(t *testing.T) {
	o := NewClient(twitchclient.ClientConf{
		Host: "foo",
	})

	err := o.InvalidateSessions(context.Background(), "12341", nil)
	assert.Equal(t, ErrInvalidRequestParams, err, "Should receive an Invalid Params error")
}

func TestInvalidateSessionsEmptyClientID(t *testing.T) {
	o := NewClient(twitchclient.ClientConf{
		Host: "foo",
	})

	req := &InvalidateSessionsRequest{
		ClientID: "",
	}
	err := o.InvalidateSessions(context.Background(), "12341", req)
	assert.Equal(t, ErrInvalidRequestParams, err, "Should receive an Invalid Params error")

}
