package sessionsclient

import (
	"context"
	"net/url"
	"strings"

	"code.justin.tv/foundation/twitchclient"
)

// ValidateSessionResponse is returned from Validate
type ValidateSessionResponse struct {
	Session *Session `json:"session"`
}

// ValidateSessionRequest is the request
type ValidateSessionRequest struct {
	ClientID string `json:"client_id"`
	Token    string `json:"token"`
}

func (c *client) Validate(ctx context.Context, validateReq *ValidateSessionRequest) (*Session, error) {
	if _, err := parseToken(validateReq.Token); err != nil {
		return nil, ErrInvalidToken
	}

	var validateResponse ValidateSessionResponse
	err := c.client.Post(ctx, "/sessions/validate", validateReq, &validateResponse)
	if err != nil {
		if e, ok := err.(*twitchclient.Error); ok {
			return nil, parseErrorResponse(e)
		}
		return nil, err
	}

	return validateResponse.Session, nil
}

func parseToken(token string) (*Session, error) {
	decodedToken, err := url.QueryUnescape(token)
	if err != nil {
		return nil, ErrInvalidToken
	}
	parts := strings.Split(decodedToken, ":")

	if len(parts) == 3 {
		return &Session{
			UserID:         parts[0],
			TokenCleartext: parts[2],
		}, nil
	}

	return nil, ErrInvalidToken
}
