package loggers

import (
	. "github.com/smartystreets/goconvey/convey"

	"bytes"
	"encoding/json"
	"fmt"
	"testing"
)

const (
	testMaintainedKey          = "someMaintainedKey"
	testMaintainedValue        = "someMaintainedValue"
	testAnotherMaintainedKey   = "someOtherMaintainedKey"
	testAnotherMaintainedValue = "someOtherMaintainedValue"
	testNilValueKey            = "someNilValue"
)

func TestDecorativeLoggerFromJSONLogger(t *testing.T) {
	Convey("Given a JSON logger", t, func() {
		byteBuffer := &bytes.Buffer{}
		encoder := json.NewEncoder(byteBuffer)
		jsonLogs := &JSONLogger{encoder, testPanic}

		Convey("And a key, value pair to maintain", func() {
			decoratingLogger := With(jsonLogs, testMaintainedKey, testMaintainedValue)

			Convey("Ensure the output has all expected information when logging a message", func() {
				decoratingLogger.Log(testMessage)

				stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
				// Verify we have a JSON string map because of our chosen logger
				So(err, ShouldBeNil)
				// Verify the content of the map
				So(stringMap, ShouldNotBeNil)
				So(stringMap, ShouldNotBeEmpty)
				So(len(stringMap), ShouldEqual, 2)
				So(stringMap[MsgKey], ShouldEqual, testMessage)
				So(stringMap[testMaintainedKey], ShouldEqual, testMaintainedValue)
			})

			Convey("Ensure the output has all expected information when logging a message with additional keyvals", func() {
				decoratingLogger.Log(testMessage, testKey, testValue)

				stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
				// Verify we have a JSON string map because of our chosen logger
				So(err, ShouldBeNil)
				// Verify the content of the map
				So(stringMap, ShouldNotBeNil)
				So(stringMap, ShouldNotBeEmpty)
				So(len(stringMap), ShouldEqual, 3)
				So(stringMap[MsgKey], ShouldEqual, testMessage)
				So(stringMap[testMaintainedKey], ShouldEqual, testMaintainedValue)
				So(stringMap[testKey], ShouldEqual, testValueString)
			})

			Convey("When adding additional valid key, value pairs to maintain", func() {
				decoratingLogger = With(decoratingLogger, testAnotherMaintainedKey, testAnotherMaintainedValue)
				Convey("Ensure the output has all expected information when logging a message", func() {
					decoratingLogger.Log(testMessage)

					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					// Verify we have a JSON string map because of our chosen logger
					So(err, ShouldBeNil)
					// Verify the content of the map
					So(stringMap, ShouldNotBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 3)
					So(stringMap[MsgKey], ShouldEqual, testMessage)
					So(stringMap[testMaintainedKey], ShouldEqual, testMaintainedValue)
					So(stringMap[testAnotherMaintainedKey], ShouldEqual, testAnotherMaintainedValue)
				})
			})

			Convey("When adding an additional key, value pair, where the value is nil", func() {
				decoratingLogger = With(decoratingLogger, testNilValueKey, nil)
				Convey("Ensure the output has all expected information when logging a message", func() {
					decoratingLogger.Log(testMessage)

					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					// Verify we have a JSON string map because of our chosen logger
					So(err, ShouldBeNil)
					// Verify the content of the map
					So(stringMap, ShouldNotBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 3)
					So(stringMap[MsgKey], ShouldEqual, testMessage)
					So(stringMap[testMaintainedKey], ShouldEqual, testMaintainedValue)
					// Although the value is nil, it should be safely converted into a string, to ensure the mapping
					// is not broken
					So(stringMap[testNilValueKey], ShouldNotBeNil)
					So(stringMap[testNilValueKey], ShouldEqual, fmt.Sprint(nil))
				})
			})
		})
	})
}

func TestDecorativeLoggerManualCreation(t *testing.T) {
	Convey("Given a Decorative logger", t, func() {
		Convey("And a wrapped JSON logger", func() {
			byteBuffer := &bytes.Buffer{}
			encoder := json.NewEncoder(byteBuffer)
			jsonLogs := &JSONLogger{encoder, testPanic}
			decoratingLogger := &Decorative{Logger: jsonLogs}
			Convey("And no key, value pairs to maintain", func() {
				Convey("Ensure the output has all expected information when logging a message", func() {
					decoratingLogger.Log(testMessage)

					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					// Verify we have a JSON string map because of our chosen logger
					So(err, ShouldBeNil)
					// Verify the content of the map
					So(stringMap, ShouldNotBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 1)
					So(stringMap[MsgKey], ShouldEqual, testMessage)
				})

				Convey("Ensure the output has all expected information when logging a message with additional keyvals", func() {
					decoratingLogger.Log(testMessage, testKey, testValue)

					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					// Verify we have a JSON string map because of our chosen logger
					So(err, ShouldBeNil)
					// Verify the content of the map
					So(stringMap, ShouldNotBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 2)
					So(stringMap[MsgKey], ShouldEqual, testMessage)
					So(stringMap[testKey], ShouldEqual, testValueString)
				})
			})
		})
	})
}

func TestDecorativeLoggerWithNilWrappedLogger(t *testing.T) {
	Convey("Given a Decorative logger", t, func() {
		Convey("With a nil logger", func() {
			decoratingLogger := &Decorative{nil, nil}
			Convey("Ensure adding keyvals does not panic", func() {
				shouldWork := func() { decoratingLogger = With(decoratingLogger, testMaintainedKey, testMaintainedValue) }
				So(shouldWork, ShouldNotPanic)
			})
			Convey("Ensure logging does not panic", func() {
				shouldWork := func() { decoratingLogger.Log(testMessage) }
				So(shouldWork, ShouldNotPanic)
			})
		})
	})
}
