package loggers

import (
	logging "code.justin.tv/amzn/TwitchLogging"
	. "github.com/smartystreets/goconvey/convey"

	"bytes"
	"encoding/json"
	"errors"
	"testing"
)

const (
	testError       = "somethingBad"
	testMessage     = "someMessage"
	testKey         = "someValue"
	testValue       = 5
	testValueString = "5"
)

func TestJSONLogger(t *testing.T) {
	Convey("Given a JSON logger", t, func() {
		Convey("With a bytes.Buffer io.Writer", func() {
			byteBuffer := &bytes.Buffer{}
			encoder := json.NewEncoder(byteBuffer)
			jsonLogs := &JSONLogger{encoder, testPanic}

			Convey("And an input with an even number of keyvals", func() {
				jsonLogs.Log(testMessage, testKey, testValue)
				Convey("Ensure the output matches the expectation", func() {
					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					// Verify we have a JSON string map (err is nil)
					So(err, ShouldBeNil)
					// Verify the content of the map
					So(stringMap, ShouldNotBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 2)
					So(stringMap[MsgKey], ShouldEqual, testMessage)
					So(stringMap[testKey], ShouldEqual, testValueString)
				})
			})

			Convey("And an input with an odd number of keyvals", func() {
				jsonLogs.Log(testMessage, testValue)
				Convey("Ensure the output matches the expectation", func() {
					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					// Verify we have a JSON string map (err is nil)
					So(err, ShouldBeNil)
					// Verify the content of the map
					So(stringMap, ShouldNotBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 2)
					So(stringMap[MsgKey], ShouldEqual, testMessage)
					So(stringMap[UnmappedKey], ShouldEqual, testValueString)
				})
			})
		})

		Convey("With a custom io.Writer that errors and an ErrorHandler that panics", func() {
			// Construct a bad io writer
			writer := &testBadIoWriter{}
			encoder := json.NewEncoder(writer)
			jsonLogs := &JSONLogger{encoder, testPanic}
			badFunction := func() { jsonLogs.Log(testMessage, testValue) }
			Convey("Ensure that the ErrorHandler panics as expected", func() {
				So(badFunction, ShouldPanicWith, testError)
			})
		})

		Convey("With a custom io.Writer that errors but no ErrorHandler", func() {
			// Construct a bad io writer
			writer := &testBadIoWriter{}
			encoder := json.NewEncoder(writer)
			jsonLogs := &JSONLogger{encoder, nil}
			badFunction := func() { jsonLogs.Log(testMessage, testValue) }
			Convey("Ensure that nothing happens since no ErrorHandler exists", func() {
				So(badFunction, ShouldNotPanic)
			})
		})
	})
}

func unmarshalJSONStringMap(bytes []byte) (map[string]string, error) {
	var stringMap map[string]string
	err := json.Unmarshal(bytes, &stringMap)
	return stringMap, err
}

// testPanicLogger represents a backup logger (ErrorHandler)
type testPanicLogger struct {
	err error
}

var testPanic ErrorHandler = &testPanicLogger{}

func (n *testPanicLogger) Log(msg string, keyvals ...interface{}) {
	panic(testError)
}

func (n *testPanicLogger) ErrorLogger(err error) logging.Logger {
	return &testPanicLogger{
		err: err,
	}
}

// testBadIoWriter represents an IO Writer that errors as soon as Write is called
type testBadIoWriter struct{}

func (n *testBadIoWriter) Write([]byte) (int, error) {
	return 0, errors.New("err message")
}
