package loggers

import (
	"errors"
	"io"
	"sync"
)

// DroppedItemCallback is an optional callback function called (from the go-routine calling Write) whenever an entry is overwritten in the buffer --i.e. when a buffered item was dropped/lost.
type DroppedItemCallback func()

//ErrCanceled is the error returned by a concurrentRingBuffer when a caller attempts to call 'write' after 'cancel'. Once a ring buffer has been canceled, no additional writes may occur.
var ErrCanceled = errors.New("buffer was canceled")

// Since writes to the underlying writer may not occur immediately, we have to copy all messages to ensure they are not modified before they are writen.
// The RingBufferWriter maintains a buffer pool to store these copies, cutting down on allocations-per-write in steady state.
var bufPool = sync.Pool{
	New: func() interface{} {
		return make([]byte, 0, 1024)
	},
}

// RingBufferWriter is a wrapper around an io.Writer that buffers all writes into a ring buffer, ensuring that all calls to Write are non-blocking.
// A RingBufferWriter ensures that output will be writen to the wrapped writer in chronological order (i.e. older entries will be written before newer entries);
// however, it makes no guarantees about whether the output is contiguous, as gaps will be created whenever the buffer wraps around and overwrites unread entries.
type RingBufferWriter struct {
	done chan bool
	buf  *concurrentRingBuffer
	w    io.Writer
}

// NewRingBufferWriter creates a new RingBufferWriter of the specified size, delegating writes to the provided Writer.
// An optional DroppedItemCallback may also be supplied, which will be called whenever the a new entry overwrites an existing entry (whenever data in the buffer is lost).
//
// It also starts a single go-routine responsible for polling the ring buffer and performing writes on the wrapped writer. The polling go-routine will run for the life of the RingBufferWriter until it is closed.
func NewRingBufferWriter(size int, w io.Writer, onDropped DroppedItemCallback) (*RingBufferWriter, error) {
	if w == nil {
		return nil, errors.New("w must be non-nil")
	}
	if size < 1 {
		return nil, errors.New("size must be >= 1")
	}

	writer := &RingBufferWriter{
		done: make(chan bool),
		buf:  newConcurrentRingBuffer(size, onDropped),
		w:    w,
	}

	go writer.poll()
	return writer, nil
}

// Write enqueues the given payload in the RingBufferWriter's internal ring buffer.
// The payload will be provided to the wrapped Writer, unless the buffer wraps around and the entry is overwritten before the entry is processed by the RingBufferWriter's polling go-routine.
func (w *RingBufferWriter) Write(p []byte) (n int, err error) {
	buf := bufPool.Get().([]byte)
	buf = append(buf, p...)

	if err := w.buf.push(buf); err != nil {
		return 0, err
	}
	return len(p), nil
}

// Close the RingBufferWriter, draining the contents of the ring buffer before returning.
// If the wrapped writer is closeable, it is also closed before this function returns.
//
// Callers should call 'Close' during a graceful shutdown to ensure messages in the buffer are written to the wrapped writer.
func (w *RingBufferWriter) Close() error {
	w.buf.cancel()
	<-w.done
	if c, ok := w.w.(io.Closer); ok {
		return c.Close()
	}
	return nil
}

func (w *RingBufferWriter) poll() {
	defer close(w.done)
	for {
		v := w.buf.poll()
		if v == nil {
			return
		}

		w.w.Write(v)
		// Pools expect all elements to be roughly the same size, ensure nothing larger than 16KiB is recycled
		// See: https://golang.org/src/fmt/print.go#L147
		if cap(v) <= 1<<14 {
			bufPool.Put(v[:0])
		}
	}
}

type concurrentRingBuffer struct {
	mut       *sync.Mutex
	cond      *sync.Cond
	onDropped DroppedItemCallback
	d         [][]byte
	readIdx   int
	writeIdx  int
	canceled  bool
}

func newConcurrentRingBuffer(size int, onDropped DroppedItemCallback) *concurrentRingBuffer {
	mut := &sync.Mutex{}
	return &concurrentRingBuffer{
		mut:       mut,
		cond:      sync.NewCond(mut),
		onDropped: onDropped,
		d:         make([][]byte, size),
	}
}

func (b *concurrentRingBuffer) push(v []byte) error {
	b.mut.Lock()
	if b.canceled {
		b.mut.Unlock()
		return ErrCanceled
	}
	if b.d[b.writeIdx] != nil {
		// To preserve the invariant that the readIdx always points to the oldest entry in the ring,
		// we must increment the readIdx if we overwrite an unread entry
		// (otherwise, it would point to the _newest_ entry in the ring)
		b.readIdx = (b.writeIdx + 1) % len(b.d)
		if b.onDropped != nil {
			b.onDropped()
		}
	}
	b.d[b.writeIdx] = v
	b.writeIdx = (b.writeIdx + 1) % len(b.d)
	b.cond.Signal()
	b.mut.Unlock()
	return nil
}

func (b *concurrentRingBuffer) poll() []byte {
	b.mut.Lock()
	for b.d[b.readIdx] == nil {
		if b.canceled {
			b.mut.Unlock()
			return nil
		}
		b.cond.Wait()
	}

	v := b.d[b.readIdx]
	b.d[b.readIdx] = nil
	b.readIdx = (b.readIdx + 1) % len(b.d)
	b.mut.Unlock()
	return v
}

func (b *concurrentRingBuffer) cancel() {
	b.mut.Lock()
	b.canceled = true
	b.cond.Signal()
	b.mut.Unlock()
}
