package logmiddleware

import (
	"bytes"
	loggers "code.justin.tv/amzn/TwitchLoggingCommonLoggers"
	"code.justin.tv/video/metricmiddleware-beta/operation"
	"context"
	"encoding/json"
	. "github.com/smartystreets/goconvey/convey"
	"testing"
	"time"
)

const (
	testServiceNameKey      = "ServiceNameKey"
	testServiceNameValue    = "TestService"
	testOperationName       = "TestOperation"
	testIgnoreOperationName = "TestIgnoreOperation"
	testDependency          = "TestDependency"
	testDependencyAPI       = "TestDependencyAPI"
	testSpan                = "TestSpan"
	testMonitorOp           = "MonitorOp"
)

func TestOperationMonitor(t *testing.T) {
	Convey("Given an OperationMonitor with a decorative logger that already has some maintained keyvals", t, func() {
		// Construct a JSON Logger
		byteBuffer := &bytes.Buffer{}
		encoder := json.NewEncoder(byteBuffer)
		jsonLogger := &loggers.JSONLogger{encoder, nil}
		// Wrap it in a decorative logger with some keyvals
		decoratingLogger := loggers.With(jsonLogger, testServiceNameKey, testServiceNameValue)
		// Construct a MonitorOp
		logOpMonitor := &OperationMonitor{
			Logger: decoratingLogger,
		}

		Convey("And given an operation starter", func() {
			operationStarter := &operation.Starter{OpMonitors: []operation.OpMonitor{logOpMonitor}}
			Convey("And a server operation", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: testOperationName}
				serverCtx, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("When successful", func() {
					serverOp.SetStatus(operation.Status{Code: 0})
					serverOp.End()
					Convey("Ensure we log the right statement", func() {
						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "server")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, testOperationName)
						So(stringMap["OpGroup"], ShouldEqual, testServiceNameKey)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "ok")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
				Convey("When failing", func() {
					serverOp.SetStatus(operation.Status{Code: 2})
					serverOp.End()
					Convey("Ensure we log the right statement", func() {
						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "server")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, testOperationName)
						So(stringMap["OpGroup"], ShouldEqual, testServiceNameKey)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "unknown")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
				Convey("And a span operation", func() {
					spanOp := operation.Name{Kind: operation.KindSpan, Group: testDependency, Method: testSpan}
					_, internalSpan := operationStarter.StartOp(serverCtx, spanOp)
					Convey("Ensure we log the right statement", func() {
						internalSpan.End()
						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "span")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, testSpan)
						So(stringMap["OpGroup"], ShouldEqual, testDependency)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "ok")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
				Convey("And a client operation", func() {
					clientOp := operation.Name{Kind: operation.KindClient, Group: testDependency, Method: testDependencyAPI}
					_, clientSpan := operationStarter.StartOp(serverCtx, clientOp)
					Convey("When successful", func() {
						Convey("Ensure we log the right statement", func() {
							clientSpan.SetStatus(operation.Status{Code: 0})
							clientSpan.End()

							stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
							So(err, ShouldBeNil)
							So(stringMap, ShouldNotBeEmpty)
							So(len(stringMap), ShouldEqual, 9)
							So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
							So(stringMap["OpKind"], ShouldEqual, "client")
							So(stringMap["OpPoint"], ShouldEqual, "End")
							So(stringMap["OpMethod"], ShouldEqual, testDependencyAPI)
							So(stringMap["OpGroup"], ShouldEqual, testDependency)
							So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
							So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
							So(stringMap["OpStatusCode"], ShouldEqual, "ok")
							So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
						})
					})
					Convey("When failing", func() {
						Convey("Ensure we log the right statement", func() {
							clientSpan.SetStatus(operation.Status{Code: 3})
							clientSpan.End()

							stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
							So(err, ShouldBeNil)
							So(stringMap, ShouldNotBeEmpty)
							So(len(stringMap), ShouldEqual, 9)
							So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
							So(stringMap["OpKind"], ShouldEqual, "client")
							So(stringMap["OpPoint"], ShouldEqual, "End")
							So(stringMap["OpMethod"], ShouldEqual, testDependencyAPI)
							So(stringMap["OpGroup"], ShouldEqual, testDependency)
							So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
							So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
							So(stringMap["OpStatusCode"], ShouldEqual, "invalid_argument")
							So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
						})
					})
				})
			})
			Convey("And a span operation", func() {
				spanOp := operation.Name{Kind: operation.KindSpan, Group: testDependency, Method: testSpan}
				_, internalSpan := operationStarter.StartOp(context.Background(), spanOp)
				Convey("Ensure we log the right statement", func() {
					internalSpan.End()

					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					So(err, ShouldBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 9)
					So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
					So(stringMap["OpKind"], ShouldEqual, "span")
					So(stringMap["OpPoint"], ShouldEqual, "End")
					So(stringMap["OpMethod"], ShouldEqual, testSpan)
					So(stringMap["OpGroup"], ShouldEqual, testDependency)
					So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
					So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
					So(stringMap["OpStatusCode"], ShouldEqual, "ok")
					So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
				})
			})
			Convey("And a client operation", func() {
				clientOp := operation.Name{Kind: operation.KindClient, Group: testDependency, Method: testDependencyAPI}
				_, clientSpan := operationStarter.StartOp(context.Background(), clientOp)
				Convey("When successful", func() {
					Convey("Ensure we log the right statement", func() {
						clientSpan.SetStatus(operation.Status{Code: 0})
						clientSpan.End()

						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "client")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, testDependencyAPI)
						So(stringMap["OpGroup"], ShouldEqual, testDependency)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "ok")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
				Convey("When failing", func() {
					Convey("Ensure we log the right statement", func() {
						clientSpan.SetStatus(operation.Status{Code: 3})
						clientSpan.End()

						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "client")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, testDependencyAPI)
						So(stringMap["OpGroup"], ShouldEqual, testDependency)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "invalid_argument")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
			})
			Convey("And an unknown server operation", func() {
				serverOpName := operation.Name{Kind: operation.Kind(-1), Group: testServiceNameValue, Method: testOperationName}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("Ensure we log the right statement", func() {
					serverOp.SetStatus(operation.Status{Code: 3})
					serverOp.End()

					stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
					So(err, ShouldBeNil)
					So(stringMap, ShouldNotBeEmpty)
					So(len(stringMap), ShouldEqual, 9)
					So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
					So(stringMap["OpKind"], ShouldEqual, "%!Kind(-1)")
					So(stringMap["OpPoint"], ShouldEqual, "End")
					So(stringMap["OpMethod"], ShouldEqual, testOperationName)
					So(stringMap["OpGroup"], ShouldEqual, testServiceNameValue)
					So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
					So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
					So(stringMap["OpStatusCode"], ShouldEqual, "invalid_argument")
					So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
				})
			})
		})
	})
}

func TestOperationMonitorWithNilLogger(t *testing.T) {
	Convey("Given an OperationMonitor with a nil logger", t, func() {
		// Construct a MonitorOp
		logOpMonitor := &OperationMonitor{
			Logger: nil,
		}
		Convey("And given an operation starter", func() {
			operationStarter := &operation.Starter{OpMonitors: []operation.OpMonitor{logOpMonitor}}
			Convey("And a server operation", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: testOperationName}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("Ensure we don't panic", func() {
					shouldWork := func() { serverOp.End() }
					So(shouldWork, ShouldNotPanic)
				})
			})
		})
	})
}

func TestOperationMonitorWithTurnOffLogging(t *testing.T) {
	Convey("Given an OperationMonitor with a decorative logger that already has some maintained keyvals but logging is disabled", t, func() {
		// Construct a JSON Logger
		byteBuffer := &bytes.Buffer{}
		encoder := json.NewEncoder(byteBuffer)
		jsonLogger := &loggers.JSONLogger{encoder, nil}
		// Wrap it in a decorative logger with some keyvals
		decoratingLogger := loggers.With(jsonLogger, testServiceNameKey, testServiceNameValue)
		// Construct a MonitorOp
		logOpMonitor := &OperationMonitor{
			Logger:         decoratingLogger,
			DisableLogging: true,
		}

		Convey("And given an operation starter", func() {
			operationStarter := &operation.Starter{OpMonitors: []operation.OpMonitor{logOpMonitor}}
			Convey("And a server operation", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: testOperationName}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("When successful", func() {
					serverOp.SetStatus(operation.Status{Code: 0})
					serverOp.End()
					Convey("Ensure we do not log", func() {
						So(byteBuffer.Bytes(), ShouldHaveLength, 0)
					})
				})
			})
		})
	})
}

func TestOperationMonitorWithIgnoreOpMethods(t *testing.T) {
	Convey("Given an OperationMonitor with a decorative logger that already has some maintained keyvals but ignores certain methods", t, func() {
		// Construct a JSON Logger
		byteBuffer := &bytes.Buffer{}
		encoder := json.NewEncoder(byteBuffer)
		jsonLogger := &loggers.JSONLogger{encoder, nil}
		// Wrap it in a decorative logger with some keyvals
		decoratingLogger := loggers.With(jsonLogger, testServiceNameKey, testServiceNameValue)
		// Construct a MonitorOp
		logOpMonitor := &OperationMonitor{
			Logger: decoratingLogger,
			MethodConfig: map[string]MethodConfig{
				testIgnoreOperationName: MethodConfig{
					DisableLogging: true,
				},
				testOperationName: MethodConfig{
					DisableLogging: false,
				},
				"Unknown": MethodConfig{
					DisableLogging: true,
				}},
		}

		Convey("And given an operation starter", func() {
			operationStarter := &operation.Starter{OpMonitors: []operation.OpMonitor{logOpMonitor}}
			Convey("And a server operation that should be logged according to IgnoreOpMethods", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: testOperationName}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("When successful", func() {
					serverOp.SetStatus(operation.Status{Code: 0})
					serverOp.End()
					Convey("Ensure we log the right statement", func() {
						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "server")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, testOperationName)
						So(stringMap["OpGroup"], ShouldEqual, testServiceNameKey)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "ok")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
			})
			Convey("And a server operation that should NOT be logged according to IgnoreOpMethods", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: testIgnoreOperationName}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("When successful", func() {
					serverOp.SetStatus(operation.Status{Code: 0})
					serverOp.End()
					Convey("Ensure we do not log", func() {
						So(byteBuffer.Bytes(), ShouldHaveLength, 0)
					})
				})
			})
			Convey("And a server operation that is not in IgnoreOpMethods", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: "someRandomOperation"}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("When successful", func() {
					serverOp.SetStatus(operation.Status{Code: 0})
					serverOp.End()
					Convey("Ensure we log the right statement", func() {
						stringMap, err := unmarshalJSONStringMap(byteBuffer.Bytes())
						So(err, ShouldBeNil)
						So(stringMap, ShouldNotBeEmpty)
						So(len(stringMap), ShouldEqual, 9)
						So(stringMap[loggers.MsgKey], ShouldEqual, testMonitorOp)
						So(stringMap["OpKind"], ShouldEqual, "server")
						So(stringMap["OpPoint"], ShouldEqual, "End")
						So(stringMap["OpMethod"], ShouldEqual, "someRandomOperation")
						So(stringMap["OpGroup"], ShouldEqual, testServiceNameKey)
						So(isValidRFC339Time(stringMap["OpStartTime"]), ShouldBeTrue)
						So(isValidRFC339Time(stringMap["OpEndTime"]), ShouldBeTrue)
						So(stringMap["OpStatusCode"], ShouldEqual, "ok")
						So(stringMap[testServiceNameKey], ShouldEqual, testServiceNameValue)
					})
				})
			})
			Convey("And a no-name operation that should NOT be logged according to IgnoreOpMethods because 'Unknown' is not allowed", func() {
				serverOpName := operation.Name{Kind: operation.KindServer, Group: testServiceNameKey, Method: ""}
				_, serverOp := operationStarter.StartOp(context.Background(), serverOpName)
				Convey("When successful", func() {
					serverOp.SetStatus(operation.Status{Code: 0})
					serverOp.End()
					Convey("Ensure we do not log", func() {
						So(byteBuffer.Bytes(), ShouldHaveLength, 0)
					})
				})
			})
		})
	})
}

func unmarshalJSONStringMap(bytes []byte) (map[string]string, error) {
	var stringMap map[string]string
	err := json.Unmarshal(bytes, &stringMap)
	return stringMap, err
}

func isValidRFC339Time(unconverted string) bool {
	_, err := time.Parse(time.RFC3339, unconverted)
	if err != nil {
		return false
	}
	return true
}
