package logmiddleware

import (
	"context"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/twitchtv/twirp"
	"testing"
)

const (
	testErrorMessage = "TestErrorMessage"
)

func TestTwirpServerHook(t *testing.T) {
	Convey("Given a Twirp server hook", t, func() {
		errorLoggingMiddleware := Server{}
		hook := errorLoggingMiddleware.ServerHooks()
		ctx := context.Background()

		Convey("With nothing configured", func() {
			Convey("When logging an error", func() {
				newCtx := hook.Error(ctx, twirp.NewError(twirp.Unknown, testErrorMessage))

				Convey("Then it should return the provided context", func() {
					So(newCtx, ShouldEqual, ctx)
				})
			})
	
			Convey("When logging nil", func() {
				newCtx := hook.Error(ctx, nil)

				Convey("Then it should return the provided context", func() {
					So(newCtx, ShouldEqual, ctx)
				})
			})
		})

		Convey("With a mock logger", func() {
			logger := &mockLogger{}
			errorLoggingMiddleware.Logger = logger

			Convey("When logging an error", func() {
				newCtx := hook.Error(ctx, twirp.NewError(twirp.Unknown, testErrorMessage))

				Convey("Then it should return the provided context", func() {
					So(newCtx, ShouldEqual, ctx)
				})
				Convey("Then it should use the configured logger", func() {
					So(logger.messages, ShouldHaveLength, 1)
					So(logger.messages[0], ShouldContainSubstring, testErrorMessage)
					So(logger.data, ShouldHaveLength, 1)
					So(logger.data[0], ShouldResemble, map[string]string{"TwirpStatusCode": "unknown"})
				})
			})

			Convey("When logging nil", func() {
				newCtx := hook.Error(ctx, nil)

				Convey("Then it should return the provided context", func() {
					So(newCtx, ShouldEqual, ctx)
				})
				Convey("Then it should not use the logger", func() {
					So(logger.messages, ShouldBeEmpty)
					So(logger.data, ShouldBeEmpty)
				})
			})

			Convey("And a suppressed status", func() {
				errorLoggingMiddleware.SuppressedErrorCodes = []twirp.ErrorCode{twirp.Unknown}

				Convey("When logging the suppressed status", func() {
					newCtx := hook.Error(ctx, twirp.NewError(twirp.Unknown, testErrorMessage))

					Convey("Then it should return the provided context", func() {
						So(newCtx, ShouldEqual, ctx)
					})
					Convey("Then it should not use the logger", func() {
						So(logger.messages, ShouldBeEmpty)
						So(logger.data, ShouldBeEmpty)
					})
				})

				Convey("When logging a status that isn't suppressed", func() {
					newCtx := hook.Error(ctx, twirp.NewError(twirp.DataLoss, testErrorMessage))

					Convey("Then it should return the provided context", func() {
						So(newCtx, ShouldEqual, ctx)
					})
					Convey("Then it should use the configured logger", func() {
						So(logger.messages, ShouldHaveLength, 1)
						So(logger.messages[0], ShouldContainSubstring, testErrorMessage)
						So(logger.data, ShouldHaveLength, 1)
						So(logger.data[0], ShouldResemble, map[string]string{"TwirpStatusCode": "data_loss"})
					})
				})
			})
		})
	})
}

type mockLogger struct {
	messages []string
	data []map[string]string
}

func (logger *mockLogger) Log(msg string, keyvals ...interface{}) {
	logger.messages = append(logger.messages, msg)

	parameters := make(map[string]string)
	for i := 0; i < len(keyvals); i += 2 {
		if i + 1 < len(keyvals) {
			parameters[keyvals[i].(string)] = keyvals[i + 1].(string)
		} else {
			parameters[keyvals[i].(string)] = "There was an odd number of parameters."
		}
	}
	logger.data = append(logger.data, parameters)
}
