package ole

import (
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchOLE/ole/internal/datakeycache"
	"code.justin.tv/amzn/TwitchOLE/ole/internal/stats"
	"github.com/aws/aws-sdk-go/service/kms"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/require"
)

func BenchmarkCryptoWithCache(b *testing.B) {
	mockController := gomock.NewController(b)
	defer mockController.Finish()

	mockKMSAPI := NewMockKMSAPI(mockController)
	encryptionKeyCache, err := datakeycache.NewCache(datakeycache.CacheConfig{
		KeyExpiration: 2 * time.Hour,
		KeyType:       datakeycache.KeyTypeEncryption,
		Reporter:      &stats.NoopReporter{},
	})
	require.NoError(b, err)

	decryptionKeyCache, err := datakeycache.NewCache(datakeycache.CacheConfig{
		KeyExpiration: 2 * time.Hour,
		KeyType:       datakeycache.KeyTypeDecryption,
		Reporter:      &stats.NoopReporter{},
	})
	require.NoError(b, err)

	client := &KMSClient{
		kms: mockKMSAPI,
		cfg: KMSOleClientConfig{
			CMKArn:    "cmk-arn",
			Algorithm: AES256GCM{},
		},
		encryptionKeyCache: encryptionKeyCache,
		decryptionKeyCache: decryptionKeyCache,
		reporter:           &stats.NoopReporter{},
	}

	plaintext := []byte("B835B7C9-87C2-4A02-AAE5-5A4A4707ACA2")
	plaintextKey := []byte("plaintextKey11111111111111111111")
	encryptedKey := []byte("encryptedKey")
	ec := map[string]string{
		"ctx": "BF5C30E2-37DF-4FA2-B3C2-96D2F42032BB",
	}

	mockKMSAPI.EXPECT().GenerateDataKey(gomock.Any()).Return(&kms.GenerateDataKeyOutput{
		Plaintext:      plaintextKey,
		CiphertextBlob: encryptedKey,
	}, nil)

	mockKMSAPI.EXPECT().Decrypt(gomock.Any()).Return(&kms.DecryptOutput{Plaintext: plaintextKey}, nil)

	// fill cache
	_, _, err = client.getOrGenerateEncryptionDataKey(ec)
	require.NoError(b, err)

	decryptionCacheKey := datakeycache.DecryptionKeyCacheCompositeKey{
		EncryptedDataKey:  encryptedKey,
		KeyProviderID:     "aws-kms",
		KeyInfo:           client.cfg.CMKArn,
		AlgorithmID:       AES256GCM{}.ID(),
		EncryptionContext: ec,
	}

	// fill cache with decryption key
	_, err = client.getOrDecryptDecryptionDataKey(decryptionCacheKey)
	require.NoError(b, err)

	encryptedObject, err := client.encrypt(plaintext, ec)
	require.NoError(b, err)

	b.ResetTimer()
	b.Run("warm/encrypt", func(b *testing.B) {
		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				obj, err := client.encrypt(plaintext, ec)
				require.NoError(b, err)
				_ = obj
			}
		})
	})

	b.Run("warm/decrypt", func(b *testing.B) {
		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				p, err := client.decrypt(encryptedObject)
				require.NoError(b, err)
				_ = p
			}
		})
	})
}
