package ole

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"log"
	"strings"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"
	metricsmiddleware "code.justin.tv/amzn/TwitchTelemetryMetricsMiddleware"
	"code.justin.tv/video/metricmiddleware-beta/awsmetric"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kms"
)

func ExampleKMSClient_NewEncryptor() {
	// Setup Telemetry for the service.
	tPid := &identifier.ProcessIdentifier{
		Service: "YourServiceName",
	}
	sender := cw.NewUnbuffered(tPid, nil)
	sampleReporter := telemetry.SampleReporter{
		SampleObserver: telemetry.NewBufferedAggregator(30*time.Second, 100000, time.Minute, sender, nil),
		SampleBuilder:  telemetry.SampleBuilder{ProcessIdentifier: *tPid},
	}

	// To get KMS request level metrics, initiate KMS session with request handler attached to the session.
	// If you do not wish to collect request level metrics, initiate session without awsmetric.Client.
	// see https://docs.fulton.twitch.a2z.com/docs/metrics.html#full-migration for details.
	metricsOpMonitor := &metricsmiddleware.OperationMonitor{
		SampleReporter: sampleReporter,
		AutoFlush:      false,
	}

	awsMetricClient := &awsmetric.Client{
		Starter: &operation.Starter{OpMonitors: []operation.OpMonitor{metricsOpMonitor}},
	}
	sess := awsMetricClient.AddToSession(
		session.Must(session.NewSession(&aws.Config{
			Region: aws.String("us-west-2"),
		})),
	)

	oleClient, err := NewKMSOleClient(kms.New(sess), KMSOleClientConfig{
		CMKArn:                "kms-key-arn",
		DataKeyTTL:            5 * time.Hour,
		EncryptionKeyMaxUsage: 10,
		Reporter:              &sampleReporter,
	})
	if err != nil {
		// config validation error
	}

	var ciphertext bytes.Buffer
	encryptionContext := map[string]string{
		"eventBusEvent": "BitsUseCreate",
		"eventBusField": "from_user_id",
	}
	encryptor := oleClient.NewEncryptor(encryptionContext, &ciphertext)
	_, err = encryptor.Write([]byte("super-secret"))
	if err != nil {
		//handler err
	}
	fmt.Printf("ciphertext: %s", ciphertext.String())
}

func ExampleKMSClient_NewDecryptor() {
	kmsClient := kms.New(
		session.Must(session.NewSession()),
		aws.NewConfig().WithRegion("us-west-2"),
	)

	oleClient, err := NewKMSOleClient(kmsClient, KMSOleClientConfig{
		CMKArn:                "kms-key-arn",
		DataKeyTTL:            5 * time.Hour,
		EncryptionKeyMaxUsage: 10,
	})
	if err != nil {
		// config validation error
	}

	ciphertext := strings.NewReader("ciphertext-generated-by-encryptor")
	decryptor := oleClient.NewDecryptor(ciphertext)
	b, err := ioutil.ReadAll(decryptor)
	if err != nil {
		log.Fatal(err)
	}
	fmt.Printf("plaintext : %s", string(b))
}
