package datakeycache

import (
	"strconv"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchOLE/ole/internal/keybytes"
	"github.com/stretchr/testify/require"
)

func BenchmarkDataKeyCache(b *testing.B) {
	// causes a cache miss every time
	newCompositeKey := func() *EncryptionKeyCacheCompositeKey {
		return &EncryptionKeyCacheCompositeKey{
			EncryptionContext: map[string]string{
				"time": strconv.FormatInt(time.Now().UnixNano(), 10),
			},
			AlgorithmID: "algorithm",
		}
	}

	b.Run("misses", func(b *testing.B) {
		cache, err := NewCache(CacheConfig{
			KeyExpiration: time.Hour,
		})
		require.NoError(b, err)

		plaintext := []byte("plaintext")
		encrypted := []byte("encrypted")

		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				key := newCompositeKey()
				p, e, err := cache.Get(key).GetOrFetchKeys(func() ([]byte, []byte, error) {
					return keybytes.Copy(plaintext), keybytes.Copy(encrypted), nil
				})
				require.NoError(b, err)
				_ = p
				_ = e
			}
		})
	})

	b.Run("hits", func(b *testing.B) {
		cache, err := NewCache(CacheConfig{
			KeyExpiration: time.Hour,
		})
		require.NoError(b, err)

		key := newCompositeKey()
		plaintext := []byte("plaintext")
		encrypted := []byte("encrypted")

		// initialize the cache
		_, _, err = cache.Get(key).GetOrFetchKeys(func() ([]byte, []byte, error) {
			return keybytes.Copy(plaintext), keybytes.Copy(encrypted), nil
		})
		require.NoError(b, err)

		b.ResetTimer()

		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				p, e, err := cache.Get(key).GetOrFetchKeys(func() ([]byte, []byte, error) {
					return keybytes.Copy(plaintext), keybytes.Copy(encrypted), nil
				})
				require.NoError(b, err)
				_ = p
				_ = e
			}
		})
	})
}
