// script to generate ../../configs_gen.go and ../../configs_gen_test.go
// This generages the Merge and Validate functions in that code base based on
// static analysis of the c7s.Config object.

package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"reflect"
	"text/template"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
)

// These attributes will not be automatically validated in the Validation
// function. See validateSpecialCases in ./configs.go for these.
var noValidationAttributes = map[string]interface{}{
	"ClientServiceURI":                        nil,
	"ClientServiceName":                       nil,
	"EnableAccessLogging":                     nil,
	"ServiceOrigins":                          nil,
	"AuthorizedDistributedIdentitiesServices": nil,
	"LogAnonymousRequestRateLimit":            nil,
}

const mergeTemplate = `// Code generated by ./internal/GenerateConfig. DO NOT EDIT.
//+build !generate

package c7s

// Merge another configuration into this one.
func (cfg *Config) Merge(other *Config) *Config{
{{range .AllAttributes}}
	if other.{{.Name}} != {{.ZeroSerialized}} {
		cfg.{{.Name}} = other.{{.Name}}
	}
{{end}}
	return cfg
}

// Validate returns an error if attributes are missing
func (cfg *Config) Validate() error {
	missing := []string{}
{{range .AllAttributes}}{{if .IncludeValidation}}
	if cfg.{{.Name}} == {{.ZeroSerialized}} {
		missing = append(missing, "{{.Name}}")
	}
{{end}}{{end}}
  if err := cfg.validateSpecialCases(); err != nil {
    if mergable, ok := err.(errMissingConfigurationMergable); ok {
      mergable.addAttributes(missing)
    }
    return err
  }

	if len(missing) > 0 {
		return newErrMissingConfiguration(missing...)
	}
	return nil
}
`

const testsTemplate = `// Code generated by ./internal/GenerateConfig. DO NOT EDIT.
//+build !generate

package c7s

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

func TestConfigMerge(t *testing.T) {
{{range .AllAttributes}}
	t.Run("{{.Name}}", func(t *testing.T) {
		original := Config{
			{{.Name}}: {{.NonZeroValueSerialized}},
		}

		t.Run("skip", func(t *testing.T) {
			cfg := new(Config)
			*cfg = original
			cfg = cfg.Merge(&Config{})
			assert.Equal(t, {{.NonZeroValueSerialized}}, cfg.{{.Name}})
		})

		t.Run("overridden", func(t *testing.T) {
			cfg := new(Config)
			*cfg = original
			cfg = cfg.Merge(&Config{
				{{.Name}}: {{.OverriddenValue}},
			})
			assert.Equal(t, {{.OverriddenValue}}, cfg.{{.Name}})
		})
	})
{{end}}
}

func filledConfiguration() Config{
	return Config{
{{range .AllAttributes}}
		{{.Name}}: {{.NonZeroValueSerialized}},
{{end}}
	}
}

func TestConfigValidate(t *testing.T) {
	filled := filledConfiguration()
{{range .AllAttributes}}{{if .IncludeValidation}}
	t.Run("{{.Name}} missing", func(t *testing.T) {
		cfg := new(Config)
		*cfg = filled
		cfg.{{.Name}} = {{.ZeroSerialized}}
		assert.Equal(t, newErrMissingConfiguration("{{.Name}}"), cfg.Validate())
	})
{{end}}{{end}}
}
`

// StructField is what is used in the templates.
type StructField struct {
	reflect.StructField
}

// IncludeValidation returns true if this attribute should be validated.
func (s StructField) IncludeValidation() bool {
	_, ok := noValidationAttributes[s.Name]
	return !ok
}

// ZeroSerialized returns the zero value of this serialized
func (s StructField) ZeroSerialized() string {
	return fmt.Sprintf("%#v", reflect.Zero(s.Type))
}

// NonZeroValueSerialized returns a string that is a non-zero value of this
// struct field.
//
// panics if the type is not supported.
func (s StructField) NonZeroValueSerialized() string {
	for _, val := range []reflect.Value{
		reflect.ValueOf("nonzero"),
		reflect.ValueOf(int64(420)),
		reflect.ValueOf(true),
		reflect.ValueOf(time.Duration(420)),
	} {
		if s.Type == val.Type() {
			return fmt.Sprintf("%s(%#v)", val.Type().String(), val)
		}
	}
	panic(fmt.Sprintf("unsupported type in configuration validation generation: %v", s.Type))
}

// OverriddenValue returns a string that is a non-zero value of this struct
// field different from NonZeroValueSerialized.
//
// panics if the type is not supported.
func (s StructField) OverriddenValue() string {
	for _, val := range []reflect.Value{
		reflect.ValueOf("overidden"),
		reflect.ValueOf(int64(620)),
		reflect.ValueOf(true),
		reflect.ValueOf(time.Duration(620)),
	} {
		if s.Type == val.Type() {
			return fmt.Sprintf("%s(%#v)", val.Type().String(), val)
		}
	}
	panic(fmt.Sprintf("unsupported type in configuration validation generation: %v", s.Type))
}

func allAttributes() []*StructField {
	e := reflect.TypeOf(&c7s.Config{}).Elem()
	attributes := make([]*StructField, 0)
	for n := 0; n < e.NumField(); n++ {
		field := e.Field(n)
		attributes = append(attributes, &StructField{StructField: field})
	}
	return attributes
}

func main() {
	output := flag.String("o", "", "output")
	templateName := flag.String("t", "merge", "template")

	flag.Parse()

	var tmpl *template.Template
	switch *templateName {
	case "merge":
		tmpl = template.Must(template.New("mergeFunction").Parse(mergeTemplate))
	case "tests":
		tmpl = template.Must(template.New("tests").Parse(testsTemplate))
	default:
		panic("invalid template: " + *templateName)
	}

	out := os.Stdout
	if *output != "" {
		fileOut, err := os.Create(*output)
		if err != nil {
			panic(err)
		}
		out = fileOut
		log.Printf("Outputing %s template to %s", *templateName, *output)
	}

	if err := tmpl.Execute(out, struct {
		AllAttributes []*StructField
	}{
		AllAttributes: allAttributes(),
	}); err != nil {
		panic(err)
	}
}
