package authorization

import (
	"encoding/json"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAudience(t *testing.T) {
	t.Run("Add", func(t *testing.T) {
		a := NewAudience()
		a.Add("https://aud1")
		assert.Equal(t, NewAudience("https://aud1"), a)
	})

	t.Run("Contains", func(t *testing.T) {
		assert.True(t, NewAudience("https://aud1").Contains("https://aud1"))
		assert.False(t, NewAudience("https://aud1").Contains("https://aud2"))
	})

	t.Run("ContainsAny", func(t *testing.T) {
		assert.True(t, NewAudience("https://aud1").ContainsAny([]string{"https://aud1"}))
		assert.False(t, NewAudience("https://aud1").ContainsAny([]string{"https://aud2"}))
		assert.True(t, NewAudience("https://aud1").ContainsAny([]string{"https://aud2", "https://aud1"}))
	})

	t.Run("String", func(t *testing.T) {
		assert.Equal(t, "[https://aud1]", NewAudience("https://aud1").String())
	})

	t.Run("json marshaling", func(t *testing.T) {
		tcs := []struct {
			Name  string
			Value *audience
		}{
			{
				Name: "as string",
				Value: &audience{
					values: map[string]interface{}{
						"https://aud1": nil,
					},
				},
			},
			{
				Name: "as array",
				Value: &audience{
					values: map[string]interface{}{
						"https://aud1": nil,
						"https://aud2": nil,
					},
				},
			},
		}

		for _, tc := range tcs {
			t.Run(tc.Name, func(t *testing.T) {
				marshaled, err := json.Marshal(tc.Value)
				require.NoError(t, err)

				var unmarshaled audience
				require.NoError(t, json.Unmarshal(marshaled, &unmarshaled))
				assert.Equal(t, tc.Value, &unmarshaled)
			})
		}
	})
}
