package authorization

import (
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAuthorization(t *testing.T) {
	t.Run("json marshaling", func(t *testing.T) {
		now := time.Now().UTC().Truncate(time.Second)

		tcs := []struct {
			Name  string
			Value *Authorization
		}{
			{
				Name: "multiple aud and scope",
				Value: &Authorization{
					Subject:    &subject{id: "my-id"},
					Audience:   NewAudience("aud1", "aud2", "aud3"),
					Scope:      token.NewScope("scope1", "scope2"),
					JWTID:      "JWTID",
					Issuer:     "ISSUER",
					Active:     true,
					NotBefore:  now.Add(-time.Hour),
					IssuedAt:   now,
					Expiration: now.Add(time.Second),
				},
			},
			{
				Name: "single aud and scope",
				Value: &Authorization{
					Subject:    &subject{id: "my-id"},
					Audience:   NewAudience("aud1"),
					Scope:      token.NewScope("scope1"),
					JWTID:      "JWTID",
					Issuer:     "ISSUER",
					Active:     true,
					NotBefore:  now.Add(-time.Hour),
					IssuedAt:   now,
					Expiration: now.Add(time.Second),
				},
			},
		}

		for _, tc := range tcs {
			marshaled, err := json.Marshal(tc.Value)
			require.NoError(t, err)

			var unmarshaled Authorization
			require.NoError(t, json.Unmarshal(marshaled, &unmarshaled))
			assert.Equal(t, tc.Value, &unmarshaled)
		}
	})
}
