package cacheauthorization

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/internal/authorization"
	"code.justin.tv/amzn/TwitchS2S2/internal/authorization/cacheauthorization/mocks"
	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestCachedAuthorizations(t *testing.T) {
	t.Run("New", func(t *testing.T) {
		authorizations := new(mocks.AuthorizationsAPI)

		assert.Equal(t,
			&CachedAuthorizations{authorizations: authorizations, cache: newLru(16)},
			New(authorizations, &c7s.Config{AccessTokenCacheSize: 16}))
	})

	t.Run("Validate", func(t *testing.T) {
		ctx := context.Background()
		const authorizationType = "AUTHORIZATIONTYPE"
		const authorizationToken = "AUTHORIZATIONTOKEN"

		now := time.Now().UTC()

		baseAuthz := func(cbs ...func(*authorization.Authorization)) *authorization.Authorization {
			authz := &authorization.Authorization{
				Subject:    authorization.NewSubject("my-id"),
				Audience:   authorization.NewAudience("aud1", "aud2", "aud3"),
				Scope:      token.NewScope("scope1", "scope2"),
				JWTID:      "JWTID",
				Issuer:     "ISSUER",
				Active:     true,
				NotBefore:  now.Add(-time.Hour),
				IssuedAt:   now,
				Expiration: now.Add(time.Hour),
			}
			for _, cb := range cbs {
				cb(authz)
			}
			return authz
		}

		onFetch := func(at *authorizationsTest) *mock.Call {
			return at.Cache.On("Fetch", authorizationType, authorizationToken).Once()
		}

		onValidate := func(at *authorizationsTest) *mock.Call {
			return at.Authorizations.On("Validate", ctx, authorizationType, authorizationToken).Once()
		}

		t.Run("from cache", func(t *testing.T) {
			t.Run("ok", func(t *testing.T) {
				at := newAuthorizationTest()
				defer at.Teardown(t)

				onFetch(at).Return(baseAuthz(), true, nil)

				authz, err := at.CachedAuthorizations.Validate(ctx, authorizationType, authorizationToken)
				require.NoError(t, err)
				assert.Equal(t, baseAuthz(), authz)
			})

			t.Run("expired", func(t *testing.T) {
				at := newAuthorizationTest()
				defer at.Teardown(t)

				onFetch(at).Return(baseAuthz(func(authz *authorization.Authorization) {
					authz.Expiration = now.Add(-time.Hour)
				}), true, nil)

				_, err := at.CachedAuthorizations.Validate(ctx, authorizationType, authorizationToken)
				authzErr, ok := err.(*authorization.ErrInvalidToken)
				require.True(t, ok)
				assert.Equal(t, authzErr.Field, "exp")
			})

			t.Run("error on fetch", func(t *testing.T) {
				at := newAuthorizationTest()
				defer at.Teardown(t)

				onFetch(at).Return(baseAuthz(func(authz *authorization.Authorization) {
					authz.Expiration = now.Add(-time.Hour)
				}), true, errors.New("error from cache"))

				_, err := at.CachedAuthorizations.Validate(ctx, authorizationType, authorizationToken)
				cacheErr := errors.New("error from cache")
				assert.Equal(t, cacheErr, err)
			})
		})

		t.Run("ok", func(t *testing.T) {
			at := newAuthorizationTest()
			defer at.Teardown(t)

			onFetch(at).Return(nil, false, nil)

			onValidate(at).Return(baseAuthz(), nil)
			at.Cache.On("Put", authorizationType, authorizationToken, baseAuthz()).Return(nil)

			authz, err := at.CachedAuthorizations.Validate(ctx, authorizationType, authorizationToken)
			require.NoError(t, err)
			assert.Equal(t, baseAuthz(), authz)
		})

		t.Run("error", func(t *testing.T) {
			at := newAuthorizationTest()
			defer at.Teardown(t)

			onFetch(at).Return(nil, false, nil)

			myErr := errors.New("myerr")
			onValidate(at).Return(nil, myErr)

			_, err := at.CachedAuthorizations.Validate(ctx, authorizationType, authorizationToken)
			assert.Equal(t, myErr, err)
		})
	})
}

func newAuthorizationTest() *authorizationsTest {
	authorizations := new(mocks.AuthorizationsAPI)
	cache := new(mocks.Cache)
	return &authorizationsTest{
		Authorizations: authorizations,
		Cache:          cache,
		CachedAuthorizations: &CachedAuthorizations{
			authorizations: authorizations,
			cache:          cache,
		},
	}
}

type authorizationsTest struct {
	Authorizations       *mocks.AuthorizationsAPI
	Cache                *mocks.Cache
	CachedAuthorizations *CachedAuthorizations
}

func (at *authorizationsTest) Teardown(t *testing.T) {
	at.Authorizations.AssertExpectations(t)
}
