package cacheauthorization

import (
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/internal/authorization"
	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"code.justin.tv/amzn/TwitchS2SJWTAlgorithms/es256"
	"code.justin.tv/sse/jwt"
	"github.com/stretchr/testify/require"
)

func BenchmarkFetch(b *testing.B) {
	blc := newBenchmarkLruCase(b)

	lru := newLru(10)

	lru.Put(blc.AuthorizationType, blc.AuthorizationToken, blc.Authorization)

	fromCacheAuthz, ok, err := lru.Fetch(blc.AuthorizationType, blc.AuthorizationToken)
	require.NoError(b, err)
	require.True(b, ok)
	require.Equal(b, blc.Authorization, fromCacheAuthz)

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			lru.Fetch(blc.AuthorizationType, blc.AuthorizationToken)
		}
	})
}

func BenchmarkPut(b *testing.B) {
	blc := newBenchmarkLruCase(b)

	lru := newLru(10)
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			lru.Put(blc.AuthorizationType, blc.AuthorizationToken, blc.Authorization)
		}
	})
}

func BenchmarkFetchAndPut(b *testing.B) {
	blc := newBenchmarkLruCase(b)

	lru := newLru(10)

	lru.Put(blc.AuthorizationType, blc.AuthorizationToken, blc.Authorization)

	fromCacheAuthz, ok, err := lru.Fetch(blc.AuthorizationType, blc.AuthorizationToken)
	require.NoError(b, err)
	require.True(b, ok)
	require.Equal(b, blc.Authorization, fromCacheAuthz)

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			lru.Put(blc.AuthorizationType, blc.AuthorizationToken, blc.Authorization)
			lru.Fetch(blc.AuthorizationType, blc.AuthorizationToken)
		}
	})
}

type benchmarkLruCase struct {
	AuthorizationType  string
	AuthorizationToken string
	Authorization      *authorization.Authorization
}

func newBenchmarkLruCase(b *testing.B) *benchmarkLruCase {
	now := time.Now().UTC().Truncate(time.Second)

	authz := &authorization.Authorization{
		Subject:    authorization.NewSubject("my-id"),
		Audience:   authorization.NewAudience("aud1", "aud2", "aud3"),
		Scope:      token.NewScope("scope1", "scope2"),
		JWTID:      "JWTID",
		Issuer:     "ISSUER",
		Active:     true,
		NotBefore:  now.Add(-time.Hour),
		IssuedAt:   now,
		Expiration: now.Add(2 * time.Hour),
	}

	key, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
	require.NoError(b, err)

	res, err := jwt.Encode(
		struct {
			KeyID     string `json:"kid"`
			Algorithm string `json:"alg"`
			Type      string `json:"typ"`
		}{
			KeyID:     "KEYID",
			Algorithm: "ES256",
			Type:      "JWT",
		},
		authz,
		&es256.PrivateKey{PrivateKey: key, KeyID: "KEYID"},
	)
	require.NoError(b, err)

	return &benchmarkLruCase{
		AuthorizationType:  string("Bearer"),
		AuthorizationToken: string(res),
		Authorization:      authz,
	}
}
