package origin

import (
	"crypto/tls"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCanonicalWebOriginFromURL(t *testing.T) {
	baseCase := "https://example.com"
	tcs := []struct {
		Input    string
		Expected string
	}{
		{Input: "https://example.com", Expected: baseCase},
		{Input: "https://example.com/", Expected: baseCase},
		{Input: "https://example.com:443", Expected: baseCase},
		{Input: "https://example.com:443/", Expected: baseCase},
		{Input: "https://example.com/path/files", Expected: baseCase},
		{Input: "https://example.com:443/path/files", Expected: baseCase},
	}

	for nTc, tc := range tcs {
		t.Run(strconv.Itoa(nTc), func(t *testing.T) {
			asURL, err := url.Parse(tc.Expected)
			require.NoError(t, err)
			assert.Equal(t, tc.Expected, CanonicalWebOriginFromURL(asURL))

			asURL, err = url.Parse(strings.ToUpper(tc.Expected))
			require.NoError(t, err)
			assert.Equal(t, tc.Expected, CanonicalWebOriginFromURL(asURL))
		})
	}
}

func TestCanonicalWebOriginFromRequestHost(t *testing.T) {
	baseCase := "https://example.com"
	tcs := []struct {
		Host     string
		TLS      *tls.ConnectionState
		Header   http.Header
		Expected string
	}{
		{Host: "example.com", TLS: &tls.ConnectionState{}, Expected: baseCase},
		{Host: "example.com:443", TLS: &tls.ConnectionState{}, Expected: baseCase},
		{Host: "example.com:80", TLS: &tls.ConnectionState{}, Expected: "https://example.com:80"},
		{Host: "example.com", Expected: "http://example.com"},
		{Host: "example.com:443", Expected: "http://example.com:443"},
		{Host: "example.com", Header: http.Header{"X-Forwarded-Proto": []string{"https"}}, Expected: baseCase},
		{Host: "", Expected: UndefinedWebOrigin},
	}

	for nTc, tc := range tcs {
		t.Run(strconv.Itoa(nTc), func(t *testing.T) {
			assert.Equal(t, tc.Expected, CanonicalWebOriginFromRequestHost(&http.Request{
				Header: tc.Header,
				Host:   tc.Host,
				TLS:    tc.TLS,
			}))

			assert.Equal(t, tc.Expected, CanonicalWebOriginFromRequestHost(&http.Request{
				Header: tc.Header,
				Host:   strings.ToUpper(tc.Host),
				TLS:    tc.TLS,
			}))
		})
	}
}
