//nolint:bodyclose
package sigv4

import (
	"errors"
	"net/http"
	"net/url"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/internal/sigv4/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestRoundTripper(t *testing.T) {
	t.Run("RoundTrip", func(t *testing.T) {
		onSign := func(rtt *roundTripperTest, req *http.Request) *mock.Call {
			return rtt.Signer.On("Sign",
				req,
				nil,
				rtt.RoundTripper.AWSService,
				rtt.RoundTripper.AWSRegion,
				mock.MatchedBy(func(st time.Time) bool {
					return time.Since(st) < time.Minute
				}))
		}

		t.Run("success", func(t *testing.T) {
			rtt := newRoundTripperTest()
			defer rtt.Teardown(t)

			req := &http.Request{URL: &url.URL{Scheme: "https", Host: "host"}}
			innerRes := &http.Response{StatusCode: http.StatusOK}

			onSign(rtt, req).Return(nil, nil).Once()
			rtt.InnerRoundTripper.On("RoundTrip", req).Return(innerRes, nil).Once()

			res, err := rtt.RoundTripper.RoundTrip(req)
			require.NoError(t, err)
			assert.Equal(t, innerRes, res)
		})

		t.Run("sign error", func(t *testing.T) {
			rtt := newRoundTripperTest()
			defer rtt.Teardown(t)

			req := &http.Request{URL: &url.URL{Scheme: "https", Host: "host"}}
			innerErr := errors.New("INNERERR")

			onSign(rtt, req).Return(nil, innerErr).Once()

			_, err := rtt.RoundTripper.RoundTrip(req)
			require.Equal(t, innerErr, err)
		})
	})
}

func newRoundTripperTest() *roundTripperTest {
	innerRoundTripper := new(mocks.RoundTripper)
	signer := new(mocks.Signer)
	return &roundTripperTest{
		RoundTripper: &RoundTripper{
			RoundTripper: innerRoundTripper,
			AWSRegion:    "AWSREGION",
			AWSService:   "AWSSERVICE",
			Signer:       signer,
		},
		InnerRoundTripper: innerRoundTripper,
		Signer:            signer,
	}
}

type roundTripperTest struct {
	RoundTripper      *RoundTripper
	InnerRoundTripper *mocks.RoundTripper
	Signer            *mocks.Signer
}

func (rtt *roundTripperTest) Teardown(t *testing.T) {
	rtt.InnerRoundTripper.AssertExpectations(t)
	rtt.Signer.AssertExpectations(t)
}
