package cachetoken

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"code.justin.tv/amzn/TwitchS2S2/internal/token/cachetoken/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCache(t *testing.T) {
	ctx := context.Background()

	t.Run("Token", func(t *testing.T) {
		const cacheKey = "CACHEKEY"
		options := token.NewOptions().WithScope(token.NewScope("myscope"))
		tok := &token.Token{
			AccessToken: "abc",
			Scope:       token.NewScope("myscope"),
		}
		t.Run("success from source then cache", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return false },
				func(*token.Options, *token.Token) bool { return false },
			)

			ct.Inner.On("Token", ctx, options).Return(tok, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)

			res, err = cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)
		})

		t.Run("success from source then stale", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return false },
			)

			ct.Inner.On("Token", ctx, options).Return(tok, nil).Twice()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)

			res, err = cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)
		})

		t.Run("success from source then expired", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return false },
				func(*token.Options, *token.Token) bool { return true },
			)

			ct.Inner.On("Token", ctx, options).Return(tok, nil).Twice()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)

			res, err = cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)
		})

		t.Run("success when stale", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return false },
			)

			cache.values[cacheKey] = &cacheValue{Token: tok}

			ct.Inner.On("Token", ctx, options).Return(nil, errors.New("TOKENERR")).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)
		})

		t.Run("success when nocache", func(t *testing.T) {

			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return false },
				func(*token.Options, *token.Token) bool { return false },
			)

			cache.values[cacheKey] = &cacheValue{Token: tok}

			newToken := &token.Token{AccessToken: "thisIsNewAccessToke"}
			options = &token.Options{
				NoCache: true,
			}
			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, newToken, res)
		})
		// If noCache is set, then we return a new value even when the scopes have not changed
		t.Run("success when scope is not changed and nocache", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return true },
			)
			cache.values[cacheKey] = &cacheValue{Token: tok, FetchTime: time.Now()}

			newToken := &token.Token{
				AccessToken: "thisIsNewAccessToke",
				Scope:       token.NewScope("newscope"),
			}
			options = &token.Options{
				NoCache: true,
			}

			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, newToken, res)
		})
		// When the scopes have changed, and nocache is not set, return the cached value
		t.Run("success when scope is changed and nocache not set", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return true },
			)
			cache.values[cacheKey] = &cacheValue{Token: tok, FetchTime: time.Now()}

			newToken := &token.Token{
				AccessToken: "thisIsNewAccessToke",
				Scope:       token.NewScope("newscope"),
			}
			options = &token.Options{
				NoCache: false,
			}

			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, newToken, res)
		})
		// If time has passed since cache refresh, and scopes have not changed,
		// update the cache and return the new token
		t.Run("success when scope is not changed from cached value but time elapsed since last update", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return true },
			)
			cache.values[cacheKey] = &cacheValue{Token: tok, FetchTime: time.Now().AddDate(-1, 0, 0)}

			newToken := &token.Token{
				AccessToken: "thisIsNewAccessToke",
				Scope:       token.NewScope("myscope"),
			}

			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, newToken, res)
		})
		// When a value exists in the cache, but the scope has changed in
		// in the new Token. Return the new Token.
		t.Run("success when scope changes from cached value", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return true },
			)
			cache.values[cacheKey] = &cacheValue{Token: tok}

			newToken := &token.Token{
				AccessToken: "thisIsNewAccessToken",
				Scope:       token.NewScope("newscope"),
			}

			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, newToken, res)
		})
		// When the new token has multiple scopes, if the cached token has a superset of those scopes
		// then the cached token is returned
		t.Run("success returning cachedToken when the cachedToken has a superset of the newToken scopes and cache is Fresh", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return true },
			)
			tok.Scope.Add("newscope")
			tok.Scope.Add("testingScope2")
			tok.Scope.Add("testingScope3")
			cache.values[cacheKey] = &cacheValue{Token: tok, FetchTime: time.Now()}

			newToken := &token.Token{
				AccessToken: "thisIsNewAccessToken",
				Scope:       token.NewScope("newscope"),
			}
			newToken.Scope.Add("testingScope2")

			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, tok, res)
		})
		// When the new token has multiple scopes, if the cached token does not have a superset of those scopes
		// then the cache is updated and the new token is returned
		t.Run("success returning newToken when the cachedToken does not have a superset of the newToken scopes", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return true },
			)
			tok.Scope.Add("testingScope2")
			cache.values[cacheKey] = &cacheValue{Token: tok}

			newToken := &token.Token{
				AccessToken: "thisIsNewAccessToken",
				Scope:       token.NewScope("newscope"),
			}
			newToken.Scope.Add("testingScope2")
			newToken.Scope.Add("testingScope3")

			ct.Inner.On("Token", ctx, options).Return(newToken, nil).Once()

			res, err := cache.Token(ctx, options)
			require.NoError(t, err)
			assert.Equal(t, newToken, res)
		})
		t.Run("error when expired", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return false },
				func(*token.Options, *token.Token) bool { return true },
			)

			tokenErr := errors.New("TOKENERR")
			ct.Inner.On("Token", ctx, options).Return(nil, tokenErr).Once()

			_, err := cache.Token(ctx, options)
			assert.Equal(t, tokenErr, err)
		})

		t.Run("fetch error", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return false },
				func(*token.Options, *token.Token) bool { return true },
			)

			tokenErr := errors.New("TOKENERR")
			ct.Inner.On("Token", ctx, options).Return(nil, tokenErr).Once()

			_, err := cache.Token(ctx, options)
			assert.Equal(t, tokenErr, err)
		})
	})

	t.Run("HardRefreshCache", func(t *testing.T) {
		const cacheKey = "CACHEKEY"
		testOptions := func(cb ...func(*token.Options)) *token.Options {
			options := token.NewOptions().WithScope(token.NewScope("myscope"))
			for _, cb := range cb {
				cb(options)
			}
			return options
		}

		testToken := func() *token.Token {
			return &token.Token{AccessToken: "abc"}
		}

		t.Run("success", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return false },
				func(*token.Options, *token.Token) bool { return false },
			)

			cache.values[cacheKey] = &cacheValue{Options: testOptions(), Token: testToken()}

			ct.Inner.On("Token", ctx,
				testOptions(func(o *token.Options) {
					o.NoCache = true
				}),
			).Return(testToken(), nil).Once()

			require.NoError(t, cache.HardRefreshCache(ctx))
		})

		t.Run("failure", func(t *testing.T) {
			ct := newCacheTest()
			defer ct.Teardown(t)

			cache := ct.NewCache(
				func(*token.Options) string { return cacheKey },
				func(*token.Options, *token.Token) bool { return true },
				func(*token.Options, *token.Token) bool { return false },
			)

			cache.values[cacheKey] = &cacheValue{Options: testOptions(), Token: testToken()}

			tokenErr := errors.New("myerr")
			ct.Inner.On("Token", ctx,
				testOptions(func(o *token.Options) {
					o.NoCache = true
				}),
			).Return(nil, tokenErr).Once()

			assert.Equal(t, tokenErr, cache.HardRefreshCache(ctx))
		})
	})
}

func newCacheTest() *cacheTest {
	return &cacheTest{Inner: new(mocks.Tokens)}
}

type cacheTest struct {
	Inner *mocks.Tokens
}

func (ct *cacheTest) Teardown(t *testing.T) {
	ct.Inner.AssertExpectations(t)
}

func (ct *cacheTest) NewCache(
	cacheKey func(*token.Options) string,
	cacheValueStale func(*token.Options, *token.Token) bool,
	cacheValueExpired func(*token.Options, *token.Token) bool,
) *Cache {
	return New(ct.Inner, cacheKey, cacheValueStale, cacheValueExpired)
}
