package s2stoken

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/internal/oidc"
	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"code.justin.tv/amzn/TwitchS2S2/internal/token/s2stoken/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestAccessTokens(t *testing.T) {
	ctx := context.Background()
	const audience = "https://audience:4838"
	const clientCredentials = "clientcredentials"

	t.Run("Tokens", func(t *testing.T) {
		accessTokenScope := token.NewScope("scope1")
		accessTokenOptions := func(cb ...func(*token.Options)) *token.Options {
			return token.NewOptions().
				WithHost(audience).
				WithScope(accessTokenScope)
		}

		onToken := func(att *accessTokensTest) *mock.Call {
			return att.ClientCredentials.On("Token", mock.Anything, token.NewOptions())
		}

		onDo := func(att *accessTokensTest) *mock.Call {
			return att.Client.
				On("Do", mock.Anything).
				Run(func(args mock.Arguments) {
					req := args.Get(0).(*http.Request)
					assert.Equal(t, "POST", req.Method)
					assert.Equal(t, &url.URL{
						Scheme: "https",
						Host:   "tokenendpoint",
						Path:   "/token",
					}, req.URL)
					att.AssertRequestFormEncodedBody(t, req, url.Values{
						"grant_type": []string{"client_credentials"},
						"scope":      []string{accessTokenScope.String()},
					})
					assert.Equal(t, http.Header{
						"Content-Type":  []string{"application/x-www-form-urlencoded"},
						"Authorization": []string{"Bearer " + clientCredentials},
						"X-Host":        []string{audience},
					}, req.Header)
				})
		}

		t.Run("success", func(t *testing.T) {
			tcs := []struct {
				TokenOptions func(*token.Options)
			}{
				{
					TokenOptions: func(*token.Options) {},
				},
				{
					TokenOptions: func(o *token.Options) {
						o.NoCache = true
					},
				},
				{
					TokenOptions: func(o *token.Options) {
						o.MustSucceed = true
					},
				},
			}

			for nTc, tc := range tcs {
				t.Run(strconv.Itoa(nTc), func(t *testing.T) {
					att := newAccessTokensTest()
					defer att.Teardown(t)

					onToken(att).Return(&token.Token{AccessToken: clientCredentials}, nil).Once()

					att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

					onDo(att).
						Return(&http.Response{
							Body: att.MarshaledJSONBody(t, &token.Token{
								AccessToken: clientCredentials,
								Scope:       accessTokenScope,
								ExpiresIn:   time.Hour,
							}),
							StatusCode: http.StatusOK,
						}, nil).Once()

					res, err := att.AccessTokens.Token(ctx, accessTokenOptions(tc.TokenOptions))
					require.NoError(t, err)
					assert.NotEmpty(t, res.Issued)
					assert.Equal(t, &token.Token{
						AccessToken: clientCredentials,
						Scope:       token.Scope{"scope1": nil},
						Issued:      res.Issued,
						ExpiresIn:   time.Hour,
					}, res)
				})
			}
		})

		t.Run("ClientCredentials.Token error", func(t *testing.T) {
			att := newAccessTokensTest()
			defer att.Teardown(t)

			tokenErr := errors.New("tokenerr")
			onToken(att).Return(nil, tokenErr).Once()

			_, err := att.AccessTokens.Token(ctx, accessTokenOptions())
			assert.Equal(t, tokenErr, err)
		})

		t.Run("Client.Do failure", func(t *testing.T) {
			att := newAccessTokensTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: clientCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			doErr := errors.New("doerr")
			onDo(att).Return(nil, doErr).Once()

			_, err := att.AccessTokens.Token(ctx, accessTokenOptions())
			assert.Equal(t, doErr, err)
		})

		t.Run("403 error", func(t *testing.T) {
			att := newAccessTokensTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: clientCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			onDo(att).
				Return(&http.Response{
					Body:       ioutil.NopCloser(bytes.NewBufferString("403 error")),
					StatusCode: http.StatusForbidden,
				}, nil).Once()

			_, err := att.AccessTokens.Token(ctx, accessTokenOptions())
			assert.IsType(t, &AccessTokensError{Message: "403 error"}, err)
		})

		t.Run("json decode error", func(t *testing.T) {
			att := newAccessTokensTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: clientCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			onDo(att).
				Return(&http.Response{
					Body:       ioutil.NopCloser(bytes.NewBufferString("}}")),
					StatusCode: http.StatusOK,
				}, nil).Once()

			_, err := att.AccessTokens.Token(ctx, accessTokenOptions())
			assert.IsType(t, &json.SyntaxError{}, err)
		})
	})
}

func newAccessTokensTest() *accessTokensTest {
	config := &c7s.Config{Issuer: "S2SISSUER"}
	client := new(mocks.HTTPClient)
	clientCredentials := new(mocks.Tokens)
	oidc := new(mocks.OIDCAPI)
	return &accessTokensTest{
		AccessTokens: &AccessTokens{
			Client:            client,
			ClientCredentials: clientCredentials,
			OIDC:              oidc,
		},
		Client:            client,
		ClientCredentials: clientCredentials,
		Config:            config,
		OIDC:              oidc,
	}
}

type accessTokensTest struct {
	AccessTokens      *AccessTokens
	Client            *mocks.HTTPClient
	ClientCredentials *mocks.Tokens
	Config            *c7s.Config
	OIDC              *mocks.OIDCAPI
}

func (att *accessTokensTest) Teardown(t *testing.T) {
	att.Client.AssertExpectations(t)
	att.ClientCredentials.AssertExpectations(t)
	att.OIDC.AssertExpectations(t)
}

func (att *accessTokensTest) AssertRequestFormEncodedBody(t *testing.T, req *http.Request, expected url.Values) {
	bs, err := ioutil.ReadAll(req.Body)
	require.NoError(t, err)
	res, err := url.ParseQuery(string(bs))
	require.NoError(t, err)
	assert.Equal(t, expected, res)
}

func (att *accessTokensTest) MarshaledJSONBody(t *testing.T, in interface{}) io.ReadCloser {
	var buf bytes.Buffer
	require.NoError(t, json.NewEncoder(&buf).Encode(in))
	return ioutil.NopCloser(&buf)
}
