package s2stoken

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/internal/oidc"
	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"code.justin.tv/amzn/TwitchS2S2/internal/token/s2stoken/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestAssertions(t *testing.T) {
	ctx := context.Background()

	t.Run("Tokens", func(t *testing.T) {
		const authorizationEndpoint = "https://my.authorization/endpoint"
		const assertionCredentials = "ASSERTIONCREDENTIALS"

		onDo := func(att *assertionsTest) *mock.Call {
			return att.Client.
				On("Do", mock.Anything).
				Run(func(args mock.Arguments) {
					req := args.Get(0).(*http.Request)
					assert.Equal(t, "GET", req.Method)
					assert.Equal(t, url.Values{
						"host":              []string{att.Config.Issuer},
						"response_type":     []string{"assertion"},
						"twitch_s2s_client": []string{att.ClientServiceURI},
					}, req.URL.Query())
					req.URL.RawQuery = ""
					assert.Equal(t, &url.URL{
						Scheme: "https",
						Host:   "my.authorization",
						Path:   "/endpoint",
					}, req.URL)
				})
		}

		t.Run("success", func(t *testing.T) {
			att := newAssertionsTest()
			defer att.Teardown(t)

			att.OIDC.On("Configuration").Return(&oidc.Configuration{
				AuthorizationEndpoint: authorizationEndpoint,
			}).Once()

			onDo(att).
				Return(&http.Response{
					Body: att.MarshaledJSONBody(t, &token.Token{
						AccessToken: assertionCredentials,
						ExpiresIn:   time.Hour,
					}),
					StatusCode: http.StatusOK,
				}, nil).Once()

			res, err := att.Assertions.Token(ctx, token.NewOptions())
			require.NoError(t, err)
			assert.NotEmpty(t, res.Issued)
			assert.Equal(t, &token.Token{
				AccessToken: assertionCredentials,
				Issued:      res.Issued,
				ExpiresIn:   time.Hour,
				Scope:       token.Scope{},
			}, res)
		})

		t.Run("do error", func(t *testing.T) {
			att := newAssertionsTest()
			defer att.Teardown(t)

			att.OIDC.On("Configuration").Return(&oidc.Configuration{
				AuthorizationEndpoint: authorizationEndpoint,
			}).Once()

			doErr := errors.New("DOERR")
			onDo(att).Return(nil, doErr).Once()

			_, err := att.Assertions.Token(ctx, token.NewOptions())
			assert.Equal(t, doErr, err)
		})

		t.Run("403 error", func(t *testing.T) {
			att := newAssertionsTest()
			defer att.Teardown(t)

			att.OIDC.On("Configuration").Return(&oidc.Configuration{
				AuthorizationEndpoint: authorizationEndpoint,
			}).Once()

			onDo(att).
				Return(&http.Response{
					Body:       ioutil.NopCloser(bytes.NewBufferString("400 error")),
					StatusCode: http.StatusForbidden,
				}, nil).Once()

			_, err := att.Assertions.Token(ctx, token.NewOptions())
			assert.IsType(t, &AssertionsError{Message: "400 error"}, err)
		})

		t.Run("decode error", func(t *testing.T) {
			att := newAssertionsTest()
			defer att.Teardown(t)

			att.OIDC.On("Configuration").Return(&oidc.Configuration{
				AuthorizationEndpoint: authorizationEndpoint,
			}).Once()

			onDo(att).
				Return(&http.Response{
					Body:       ioutil.NopCloser(bytes.NewBufferString("}}")),
					StatusCode: http.StatusOK,
				}, nil).Once()

			_, err := att.Assertions.Token(ctx, token.NewOptions())
			assert.IsType(t, &json.SyntaxError{}, err)
		})
	})
}

func newAssertionsTest() *assertionsTest {
	const clientServiceURI = "CLIENTSERVICEURI"
	config := &c7s.Config{
		ClientServiceURI: "clientserviceuri",
		Issuer:           "myissuer",
	}
	client := new(mocks.HTTPClient)
	clientCredentials := new(mocks.Tokens)
	oidc := new(mocks.OIDCAPI)
	return &assertionsTest{
		Assertions: &Assertions{
			ClientServiceURI: clientServiceURI,
			Config:           config,
			OIDC:             oidc,
			SigV4Client:      client,
		},
		Client:            client,
		ClientCredentials: clientCredentials,
		ClientServiceURI:  clientServiceURI,
		Config:            config,
		OIDC:              oidc,
	}
}

type assertionsTest struct {
	Assertions        *Assertions
	Client            *mocks.HTTPClient
	ClientCredentials *mocks.Tokens
	ClientServiceURI  string
	Config            *c7s.Config
	OIDC              *mocks.OIDCAPI
}

func (att *assertionsTest) Teardown(t *testing.T) {
	att.Client.AssertExpectations(t)
	att.ClientCredentials.AssertExpectations(t)
	att.OIDC.AssertExpectations(t)
}

func (att *assertionsTest) AssertRequestFormEncodedBody(t *testing.T, req *http.Request, expected url.Values) {
	bs, err := ioutil.ReadAll(req.Body)
	require.NoError(t, err)
	res, err := url.ParseQuery(string(bs))
	require.NoError(t, err)
	assert.Equal(t, expected, res)
}

func (att *assertionsTest) MarshaledJSONBody(t *testing.T, in interface{}) io.ReadCloser {
	var buf bytes.Buffer
	require.NoError(t, json.NewEncoder(&buf).Encode(in))
	return ioutil.NopCloser(&buf)
}
