package s2stoken

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/internal/oidc"
	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"code.justin.tv/amzn/TwitchS2S2/internal/token/s2stoken/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestClientCredentials(t *testing.T) {
	ctx := context.Background()
	const assertionCredentials = "ASSERTIONCREDENTIALS"
	const clientCredentials = "CLIENTCREDENTIALS"

	t.Run("Tokens", func(t *testing.T) {
		onToken := func(att *clientCredentialsTest) *mock.Call {
			return att.Assertions.On("Token", mock.Anything, token.NewOptions())
		}

		onDo := func(att *clientCredentialsTest) *mock.Call {
			return att.Client.
				On("Do", mock.Anything).
				Run(func(args mock.Arguments) {
					req := args.Get(0).(*http.Request)
					assert.Equal(t, "POST", req.Method)
					assert.Equal(t, &url.URL{
						Scheme: "https",
						Host:   "tokenendpoint",
						Path:   "/token",
					}, req.URL)
					att.AssertRequestFormEncodedBody(t, req, url.Values{
						"grant_type": []string{"twitch_s2s_service_credentials"},
						"assertion":  []string{assertionCredentials},
						"scope":      []string{att.Config.TokenScope},
					})
					assert.Equal(t, http.Header{
						"Content-Type": []string{"application/x-www-form-urlencoded"},
						"X-Host":       []string{att.Config.Issuer},
					}, req.Header)
				})
		}

		t.Run("success", func(t *testing.T) {
			att := newClientCredentialsTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: assertionCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			onDo(att).
				Return(&http.Response{
					Body: att.MarshaledJSONBody(t, &token.Token{
						AccessToken: clientCredentials,
						Scope:       token.Scope{"scope1": nil},
						ExpiresIn:   time.Hour,
					}),
					StatusCode: http.StatusOK,
				}, nil).Once()

			res, err := att.ClientCredentials.Token(ctx, token.NewOptions())
			require.NoError(t, err)
			assert.NotEmpty(t, res.Issued)
			assert.Equal(t, &token.Token{
				AccessToken: clientCredentials,
				Scope:       token.Scope{"scope1": nil},
				Issued:      res.Issued,
				ExpiresIn:   time.Hour,
			}, res)
		})

		t.Run("ClientCredentials.Token error", func(t *testing.T) {
			att := newClientCredentialsTest()
			defer att.Teardown(t)

			tokenErr := errors.New("tokenerr")
			onToken(att).Return(nil, tokenErr).Once()

			_, err := att.ClientCredentials.Token(ctx, token.NewOptions())
			assert.Equal(t, tokenErr, err)
		})

		t.Run("Client.Do failure", func(t *testing.T) {
			att := newClientCredentialsTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: assertionCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			doErr := errors.New("doerr")
			onDo(att).Return(nil, doErr).Once()

			_, err := att.ClientCredentials.Token(ctx, token.NewOptions())
			assert.Equal(t, doErr, err)
		})

		t.Run("403 error", func(t *testing.T) {
			att := newClientCredentialsTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: assertionCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			onDo(att).
				Return(&http.Response{
					Body:       ioutil.NopCloser(bytes.NewBufferString("myerror")),
					StatusCode: http.StatusForbidden,
				}, nil).Once()

			_, err := att.ClientCredentials.Token(ctx, token.NewOptions())
			assert.Equal(t, &ClientCredentialsError{Message: "myerror"}, err)
		})

		t.Run("json decode error", func(t *testing.T) {
			att := newClientCredentialsTest()
			defer att.Teardown(t)

			onToken(att).Return(&token.Token{AccessToken: assertionCredentials}, nil).Once()

			att.OIDC.On("Configuration").Return(&oidc.Configuration{TokenEndpoint: "https://tokenendpoint/token"}).Once()

			onDo(att).
				Return(&http.Response{
					Body:       ioutil.NopCloser(bytes.NewBufferString("}}")),
					StatusCode: http.StatusOK,
				}, nil).Once()

			_, err := att.ClientCredentials.Token(ctx, token.NewOptions())
			assert.IsType(t, &json.SyntaxError{}, err)
		})
	})
}

func newClientCredentialsTest() *clientCredentialsTest {
	cfg := &c7s.Config{TokenScope: "S2STOKENSCOPE", Issuer: "S2SISSUER"}
	client := new(mocks.HTTPClient)
	assertions := new(mocks.Tokens)
	oidc := new(mocks.OIDCAPI)
	return &clientCredentialsTest{
		ClientCredentials: &ClientCredentials{
			Assertions: assertions,
			Client:     client,
			Config:     cfg,
			OIDC:       oidc,
		},
		Config:     cfg,
		Client:     client,
		Assertions: assertions,
		OIDC:       oidc,
	}
}

type clientCredentialsTest struct {
	ClientCredentials *ClientCredentials
	Config            *c7s.Config
	Client            *mocks.HTTPClient
	Assertions        *mocks.Tokens
	OIDC              *mocks.OIDCAPI
}

func (att *clientCredentialsTest) Teardown(t *testing.T) {
	att.Client.AssertExpectations(t)
	att.Assertions.AssertExpectations(t)
	att.OIDC.AssertExpectations(t)
}

func (att *clientCredentialsTest) AssertRequestFormEncodedBody(t *testing.T, req *http.Request, expected url.Values) {
	bs, err := ioutil.ReadAll(req.Body)
	require.NoError(t, err)
	res, err := url.ParseQuery(string(bs))
	require.NoError(t, err)
	assert.Equal(t, expected, res)
}

func (att *clientCredentialsTest) MarshaledJSONBody(t *testing.T, in interface{}) io.ReadCloser {
	var buf bytes.Buffer
	require.NoError(t, json.NewEncoder(&buf).Encode(in))
	return ioutil.NopCloser(&buf)
}
