package token

import (
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestScope(t *testing.T) {
	t.Run("NewScope", func(t *testing.T) {
		assert.Equal(t, Scope{"scope1": nil}, NewScope("scope1"))
	})

	t.Run("ParseScope", func(t *testing.T) {
		t.Run("empty string", func(t *testing.T) {
			assert.Equal(t, Scope{}, ParseScope(""))
		})

		t.Run("len(1)", func(t *testing.T) {
			assert.Equal(t, Scope{"scope1": nil}, ParseScope("scope1"))
		})

		t.Run("len(2)", func(t *testing.T) {
			assert.Equal(t, Scope{"scope1": nil, "scope2": nil}, ParseScope("scope1 scope2"))
		})

		t.Run("invalid path escaped", func(t *testing.T) {
			assert.Equal(t, Scope{"%A": nil}, ParseScope("%A"))
		})
	})

	t.Run("Add", func(t *testing.T) {
		s := make(Scope)
		s.Add("scope1")
		assert.Equal(t, Scope{"scope1": nil}, s)
	})

	t.Run("Contains", func(t *testing.T) {
		t.Run("exact", func(t *testing.T) {
			assert.True(t, NewScope("scope1").Contains(NewScope("scope1")))
		})

		t.Run("subset", func(t *testing.T) {
			assert.True(t, NewScope("s1", "s2").Contains(NewScope("s1")))
		})

		t.Run("only intersection", func(t *testing.T) {
			assert.False(t, NewScope("s1", "s2").Contains(NewScope("s1", "s3")))
		})

		t.Run("no intersection", func(t *testing.T) {
			assert.False(t, NewScope("s1", "s2").Contains(NewScope("s3")))
		})
	})

	t.Run("IsSuperSetOf", func(t *testing.T) {
		t.Run("exact", func(t *testing.T) {
			assert.True(t, NewScope("scope1").IsSuperSetOf([]string{"scope1"}))
		})

		t.Run("subset", func(t *testing.T) {
			assert.True(t, NewScope("s1", "s2").IsSuperSetOf([]string{"s1"}))
		})

		t.Run("only intersection", func(t *testing.T) {
			assert.False(t, NewScope("s1", "s2").IsSuperSetOf([]string{"s1", "s3"}))
		})

		t.Run("no intersection", func(t *testing.T) {
			assert.False(t, NewScope("s1", "s2").IsSuperSetOf([]string{"s3"}))
		})
	})

	t.Run("HasAnyAuthorization", func(t *testing.T) {
		tcs := []struct {
			Name             string
			Scope            string
			ClientServiceURI string
			Expected         bool
		}{
			{
				Name:             "is contained",
				Scope:            "https://myauthority/myservice#something",
				ClientServiceURI: "https://myauthority/myservice",
				Expected:         true,
			},
			{
				Name:             "is not contained",
				Scope:            "https://myauthority/myservice1#something",
				ClientServiceURI: "https://myauthority/myservice2",
				Expected:         false,
			},
		}

		for _, tc := range tcs {
			t.Run(tc.Name, func(t *testing.T) {
				assert.Equal(t, tc.Expected, NewScope(tc.Scope).HasAnyAuthorization(tc.ClientServiceURI))
			})
		}
	})

	t.Run("Extend", func(t *testing.T) {
		assert.Equal(t, NewScope("s1", "s2"), NewScope("s1").Extend(NewScope("s2")))
	})

	t.Run("String", func(t *testing.T) {
		t.Run("single", func(t *testing.T) {
			assert.Equal(t, "s1", NewScope("s1").String())
		})

		t.Run("double", func(t *testing.T) {
			assert.ElementsMatch(t, []string{"s1", "s2"}, strings.Split(NewScope("s1", "s2").String(), " "))
		})

		t.Run("single url encoded", func(t *testing.T) {
			assert.Equal(t, "abc%20def", NewScope("abc def").String())
		})
	})
}
